/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.models;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.openexchange.mobile.api.facade.connectors.responses.mtos.AttachmentMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.MailMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.ThreadReferenceMailMto;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MailContactUtil;
import com.openexchange.mobile.api.facade.utils.MailUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.Data;
import lombok.NonNull;

@Data
public class Mail {

    private static final MapFunction<AttachmentMto, Attachment> partMapFunction = new MapFunction<AttachmentMto, Attachment>() {
        @Override
        public Attachment map(AttachmentMto attachmentMto) {
            return new Attachment(attachmentMto);
        }
    };

    // Properties stubs

    @NonNull
    private final String id;

    private String rootMailId;

    private String folderId;

    private final MailFlags flags;

    private final Integer color;

    private final Integer level;

    @NonNull
    private final Long receivedDate;

    private final Long sentDate;

    // Properties envelopes

    private final List<MailContact> from;

    private final List<MailContact> to;

    private final List<MailContact> cc;

    private final List<MailContact> bcc;

    private final String subject;

    // Properties mails

    private String teaser;

    private final String messageId;

    private final List<MailContact> replyTo;

    private final String accountName;

    private final Integer accountId;

    private final String contentType;

    private final Boolean hasAttachments;

    private final Long size;

    private final Priority priority;

    private List<Attachment> parts;

    private Mail(String id, int flags, int color, Long receivedDate, int level) {
        this.id = id;
        this.flags = new MailFlags(flags);
        this.color = nonZero(color);
        this.level = level;
        this.receivedDate = receivedDate;
        this.sentDate = null;
        this.from = null;
        this.to = null;
        this.cc = null;
        this.bcc = null;
        this.subject = null;
        this.replyTo = null;
        this.messageId = null;
        this.accountName = null;
        this.accountId = null;
        this.contentType = null;
        this.hasAttachments = null;
        this.size = null;
        this.priority = null;
        this.parts = null;
    }

    private Mail(String id, int flags, int color, Long receivedDate, List<MailContact> from, List<MailContact> to, List<MailContact> cc, List<MailContact> bcc, String subject, String teaser, Long sentDate, Integer priority, Boolean hasAttachments) {
        this.id = id;
        this.flags = new MailFlags(flags);
        this.color = nonZero(color);
        this.level = null;
        this.receivedDate = receivedDate;
        this.sentDate = sentDate;
        this.from = from;
        this.to = to;
        this.cc = cc;
        this.bcc = bcc;
        this.subject = subject;
        this.teaser = teaser;
        this.replyTo = null;
        this.messageId = null;
        this.accountName = null;
        this.accountId = null;
        this.contentType = null;
        this.hasAttachments = hasAttachments;
        this.size = null;
        this.priority = Priority.valueOf(priority);
        this.parts = null;
    }

    private Mail(MailMto mailMto) {
        this.id = mailMto.getId();
        this.flags = new MailFlags(mailMto.getFlags());
        this.color = nonZero(mailMto.getColor());
        this.level = null;
        this.receivedDate = mailMto.getReceivedDate();
        this.sentDate = mailMto.getSentDate();
        this.from = MailContactUtil.convert(mailMto.getFrom());
        this.to = MailContactUtil.convert(mailMto.getTo());
        this.cc = MailContactUtil.convert(mailMto.getCc());
        this.bcc = MailContactUtil.convert(mailMto.getBcc());
        this.subject = mailMto.getSubject();
        Map<String, Object> headers = getHeaders(mailMto);
        this.messageId = getString(headers.get("Message-ID"));
        this.replyTo = MailUtil.parseContacts((String) headers.get("Reply-To"));
        this.accountName = mailMto.getAccountName();
        this.accountId = mailMto.getAccountId();
        this.contentType = mailMto.getContentType();
        this.hasAttachments = mailMto.isRealAttachment() ? Boolean.TRUE : null;
        this.size = mailMto.getSize();
        this.priority = priorityOrNull(mailMto.getPriority());
        this.parts = ListUtil.map(mailMto.getAttachments(), partMapFunction);
    }

    public static Mail createStub(MailMto mailMto) {
        return new Mail(mailMto.getId(), mailMto.getFlags(), mailMto.getColor(), mailMto.getReceivedDate(), mailMto.getLevel());
    }

    public static Mail createStub(ThreadReferenceMailMto mailMto) {
        return new Mail(mailMto.getId(), mailMto.getFlags(), mailMto.getColor(), mailMto.getReceivedDate(), 0);
    }

    public static Mail createStub(String id, int flags, int color, Long receivedDate, int level) {
        return new Mail(id, flags, color, receivedDate, level);
    }

    public static Mail createEnvelope(String id, int flags, int color, Long receivedDate, List<MailContact> from, List<MailContact> to, List<MailContact> cc, List<MailContact> bcc, String subject, String teaser, Long sentDate, Integer priority, Boolean hasAttachments) {
        return new Mail(id, flags, color, receivedDate, from, to, cc, bcc, subject, teaser, sentDate, priority, hasAttachments);
    }

    public static Mail createMail(MailMto mailMto) {
        return new Mail(mailMto);
    }

    private Map<String, Object> getHeaders(MailMto mailMto) {
        Map<String, Object> headers = mailMto.getHeaders();
        return headers != null ? headers : Collections.<String, Object>emptyMap();
    }

    private String getString(Object object) {
        if (object instanceof String) {
            return (String) object;
        } else if (object instanceof List) {
            return (String) ((List<?>) object).get(0);
        } else if (object == null) {
            return null;
        } else {
            throw new RuntimeException("Unsupported type to map to String: " + object.getClass().getSimpleName());
        }
    }

    private Integer nonZero(int value) {
        return value == 0 ? null : value;
    }

    private Priority priorityOrNull(int value) {
        return value != 0 ? Priority.valueOf(value) : null;
    }

}
