/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.models;

import java.util.List;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.AccountMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.UserMto;
import com.openexchange.mobile.api.facade.utils.AccountUtil;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.Data;
import lombok.NonNull;

@Data
public class Account {

    @NonNull
    private final String id;

    @NonNull
    private final String accountName;

    private final String userName;

    private final String firstName;

    private final String lastName;

    @NonNull
    private final String primaryAddress;

    private final List<String> aliases;

    private final boolean canSend;

    private final String rootFolderId;

    private final String archiveFolderId;

    private List<Folder> folders;

    private StandardFolders standardFolders;

    private final boolean pop3Account;

    public Account(AccountMto accountMto, UserMto userMto, String defaultSendAddress) {
        this(accountMto, userMto, defaultSendAddress, null);
    }

    public Account(AccountMto accountMto, UserMto userMto, String defaultSendAddress, List<Folder> folders) {
        id = accountMto.getId();
        accountName = accountMto.getAccountName();
        userName = AccountUtil.getUserName(accountMto.getPersonalName(), userMto);
        if (userMto != null) {
            firstName = userMto.getFirstName();
            lastName = userMto.getLastName();
        } else {
            firstName = null;
            lastName = null;
        }
        String accountPrimaryAddress = accountMto.getPrimaryAddress();
        if (defaultSendAddress == null || accountPrimaryAddress.equals(defaultSendAddress)) {
            primaryAddress = accountPrimaryAddress;
            aliases = accountMto.getAliases();
        } else {
            primaryAddress = defaultSendAddress;
            aliases = replaceAddress(accountMto.getAliases(), defaultSendAddress, accountPrimaryAddress);
        }
        canSend = "0".equals(accountMto.getId()) || !Strings.isEmpty(accountMto.getTransportUrl());
        rootFolderId = accountMto.getRootFolderId();
        archiveFolderId = accountMto.getArchiveFolderId();
        if (folders != null) {
            /* Java8: Use with Java 8.
            List<FolderCto> folders = folderMtos.stream()
                    .map(folderMto -> new FolderCto(folderMto.getId(), folderMto.getParentId(), folderMto.getName(), 0, 0, 0, 0))
                    .collect(Collectors.toList());
            */
            this.folders = folders;
            standardFolders = createStandardFolders(folders);
        }
        pop3Account = accountMto.isPop3Account();
    }

    public Account withFolders(List<Folder> folders) {
        this.folders = folders;
        standardFolders = createStandardFolders(folders);
        return this;
    }

    private StandardFolders createStandardFolders(List<Folder> folders) {
        Folder inboxFolder = null;
        Folder archiveFolder = null;
        Folder draftsFolder = null;
        Folder sentFolder = null;
        Folder spamFolder = null;
        Folder trashFolder = null;

        for (Folder folder : folders) {
            switch (folder.getType()) {
            case INBOX:
                inboxFolder = folder;
                break;
            case ARCHIVE:
                archiveFolder = folder;
                break;
            case DRAFT:
                draftsFolder = folder;
                break;
            case SENT:
                sentFolder = folder;
                break;
            case SPAM:
                spamFolder = folder;
                break;
            case TRASH:
                trashFolder = folder;
                break;
            default:
                // Nothing to do.
                break;
            }
        }

        return new StandardFolders(draftsFolder, inboxFolder, sentFolder, spamFolder, trashFolder, archiveFolder);
    }

    private List<String> replaceAddress(List<String> addresses, final String toReplace, final String replacement) {
        return ListUtil.map(addresses, new MapFunction<String, String>() {

            @Override
            public String map(String address) {
                return address.equals(toReplace) ? replacement : address;
            }

        });
    }

    public boolean isPrimaryAccount() {
        return "0".equals(id);
    }

}
