/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import java.net.URI;
import java.util.regex.Pattern;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.ResponseBuilder;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.AuthenticationService;
import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.configuration.ConfigurationHelper;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.endpoints.requests.AuthenticationRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.LogoutRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.PushRequestBody;
import com.openexchange.mobile.api.facade.endpoints.responses.AuthenticationResponseBody;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.osgi.MultipleAuthenticationServicesHandler;
import com.openexchange.mobile.api.facade.services.ConfigService;
import com.openexchange.mobile.api.facade.utils.CookieUtil;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;
import com.openexchange.mobile.api.facade.utils.UserAgentInfo;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpStatus;

@Path("/api-facade/v1/auth")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class AuthenticationEndpoint {

    private static final String FIELD_TOKEN = "token";

    private static final String FIELD_TRANSPORT = "transport";

    private static final Pattern PATTERN_CLIENT_ID = Pattern.compile("^\\p{Alnum}*$");

    private final MultipleAuthenticationServicesHandler authenticationHandler;

    private final ConfigService configService;

    @POST
    @Path("/login")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response authenticate(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Authenticating client");
        AuthenticationRequestBody authenticationRequestBody = JacksonUtil.parse(rawRequestBody, AuthenticationRequestBody.class);
        validate(authenticationRequestBody);
        String hostname = RequestUtil.getHostHeader(request);
        String xHostname = RequestUtil.getXHostHeader(request);
        String userAgent = RequestUtil.getUserAgent(request);
        String forwardedFor = RequestUtil.getForwardedForWithRemoteIp(request);
        String proto = RequestUtil.getForwardedProto(request);
        String trackingId = RequestUtil.getTrackingId(request);
        UserAgentInfo userAgentInfo = RequestUtil.parseUserAgent(userAgent);
        HostConfiguration configuration = ConfigurationHelper.getConfiguration(hostname, xHostname, userAgentInfo);
        RequestUtil.checkUserAgent(userAgentInfo, configuration);
        LoginCredentials credentials = getCredentials(request, authenticationRequestBody)
                .withHostConfiguration(configuration)
                .withHost(hostname)
                .withXHost(xHostname)
                .withUserAgent(userAgent)
                .withForwardedFor(forwardedFor)
                .withForwardedProto(proto)
                .withTrackingId(trackingId);
        try {
            AuthenticationService authenticationService = authenticationHandler.getAuthenticationService(configuration);
            SessionData sessionData = authenticationService.authenticate(credentials);
            sessionData = RequestUtil.enhanceSessionData(request, configuration, hostname, xHostname, userAgent, null, null, null, sessionData);
            sessionData = configService.addConfiguration(sessionData);
            return buildResponse(sessionData);
        } catch (ApiFacadeException e) {
            if (HttpStatus.SC_TEMPORARY_REDIRECT == e.getResponse().getStatus()) {
                return buildRedirectResponse(e);
            }
            throw e;
        }
    }

    void validate(AuthenticationRequestBody authenticationRequestBody) {
        if (authenticationRequestBody == null) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.INTERNAL_SERVER_ERROR, "Request body missing");
        }
        String clientId = authenticationRequestBody.getClientId();
        if (clientId != null && !PATTERN_CLIENT_ID.matcher(clientId).matches()) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.INTERNAL_SERVER_ERROR, "Invalid client ID");
        }
    }

    private LoginCredentials getCredentials(HttpServletRequest request, AuthenticationRequestBody requestBody) {
        String username = requestBody.getUsername();
        String password = requestBody.getPassword();
        String clientId = requestBody.getClientId();
        if (Strings.isEmpty(username)) {
            throw ApiFacadeException.missingParameter("username");
        }
        if (Strings.isEmpty(password)) {
            throw ApiFacadeException.missingParameter("password");
        }
        if ("".equals(clientId)) {
            clientId = null;
        }
        return new LoginCredentials(username, password, clientId)
                .withForwardedFor(RequestUtil.getForwardedForWithRemoteIp(request))
                .withRemoteIp(RequestUtil.getRemoteIp(request));
    }

    @POST
    @Path("/logout")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response logout(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Logging out client");
        HostConfiguration configuration = sessionData.getConfiguration();
        AuthenticationService authenticationService = authenticationHandler.getAuthenticationService(configuration);
        LogoutRequestBody requestBody = Strings.isEmpty(rawRequestBody)
                ? new LogoutRequestBody(null)
                : JacksonUtil.parse(rawRequestBody, LogoutRequestBody.class);
        validateLogoutRequestBody(requestBody);
        authenticationService.logout(sessionData, requestBody);
        return RequestUtil.emptyJsonResponse();
    }

    private void validateLogoutRequestBody(LogoutRequestBody requestBody) {
        PushRequestBody pushInfo = requestBody.getPushInfo();
        if (pushInfo == null) {
            return;
        }
        RequestUtil.checkParam(FIELD_TOKEN, pushInfo.getToken());
        RequestUtil.checkParam(FIELD_TRANSPORT, pushInfo.getTransport());
    }

    private Response buildResponse(SessionData sessionData) {
        AuthenticationResponseBody responseBody = new AuthenticationResponseBody(sessionData);
        String rawResponseBody = JacksonUtil.toString(responseBody);
        Response.ResponseBuilder builder = Response.ok(rawResponseBody);
        CookieUtil.addToResponse(builder, sessionData.getCookies());
        return RequestUtil.nonCachable(builder).build();
    }

    @SneakyThrows
    private Response buildRedirectResponse(ApiFacadeException e) {
        String baseUrl = e.getMessage();
        ResponseBuilder builder = Response.temporaryRedirect(new URI(baseUrl));
        return RequestUtil.nonCachable(builder).build();
    }

}
