/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.connectors;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.CookieStore;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.cookie.BasicClientCookie;
import org.apache.http.util.EntityUtils;

@Slf4j
public abstract class AbstractConnector<T> implements Connector<T, T> {

    private final RequestConfiguration configuration;

    protected final SessionData sessionData;

    private final String module;

    public AbstractConnector(RequestConfiguration configuration, String module) {
        this(configuration, null, module);
    }

    public AbstractConnector(RequestConfiguration configuration, SessionData sessionData, String module) {
        this.configuration = configuration;
        this.sessionData = sessionData;
        this.module = module;
    }

    protected abstract HttpUriRequest createRequest();

    protected void addSessionData(RequestBuilder builder) {
        HostConfiguration configuration = sessionData.getConfiguration();
        if (configuration.isForwardHostHeader()) {
            String hostname = sessionData.getHost();
            String xHostname = sessionData.getXHost();
            if (!Strings.isEmpty(hostname)) {
                builder.addHeader(RequestUtil.HEADER_HOST, hostname);
            }
            if (!Strings.isEmpty(xHostname)) {
                builder.addHeader(RequestUtil.HEADER_X_HOST, xHostname);
            }
        }
        String forwardedFor = sessionData.getForwardedFor();
        if (!Strings.isEmpty(forwardedFor)) {
            builder.addHeader(RequestUtil.HEADER_X_FORWARDED_FOR, forwardedFor);
        }
        String userAgent = sessionData.getUserAgent();
        if (!Strings.isEmpty(userAgent)) {
            builder.addHeader(RequestUtil.HEADER_USER_AGENT, userAgent);
        }
        builder.addParameter("session", sessionData.getSession());
        String trackingId = sessionData.getProperty(SessionData.PROPERTY_TRACKING_ID);
        if (!Strings.isEmpty(trackingId)) {
            builder.addParameter("trackingId", trackingId);
        }
    }

    @Override
    public String createUrl() {
        return configuration.getBaseUrl() + module;
    }

    @Override
    public T execute() {
        CloseableHttpClient httpClient = configuration.getHttpClient();
        HttpUriRequest request = createRequest();
        HttpClientContext context = HttpClientContext.create();
        if (sessionData != null) {
            CookieStore cookieStore = configuration.getCookieStore();
            for (Cookie cookie : sessionData.getCookies()) {
                BasicClientCookie basicCookie = (BasicClientCookie) cookie;
                basicCookie.setDomain(request.getURI().getHost());
                basicCookie.setSecure(false);
                cookieStore.addCookie(basicCookie);
            }
            context.setCookieStore(cookieStore);
        }
        try (CloseableHttpResponse response = httpClient.execute(request, context)) {
            return handleResponse(response);
        } catch (Exception e) {
            log.error("Failed to execute request", e);
            if (e instanceof ConnectTimeoutException) {
                throw ApiFacadeException.gatewayTimeoutError(e);
            }
            throw ApiFacadeException.internalServerError(e);
        }
    }

    @Override
    public T handleResponse(HttpResponse response) throws UnsupportedOperationException {
        try {
            StatusLine statusLine = response.getStatusLine();
            int statusCode = statusLine.getStatusCode();
            HttpEntity entity = response.getEntity();
            ObjectMapper mapper = JacksonUtil.getMapper();
            T responseBody;
            JavaType responseBodyType = getResponseBodyType(mapper);
            if (log.isDebugEnabled()) {
                entity = new BufferedHttpEntity(response.getEntity());
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                entity.writeTo(baos);
                log.debug(new String(baos.toByteArray(), "UTF-8"));
            }
            try {
                responseBody = mapper.readValue(entity.getContent(), responseBodyType);
            } catch (JsonMappingException e) {
                JavaType errorResponseBodyType = mapper.getTypeFactory().constructType(AbstractResponseMto.class);
                AbstractResponseMto errorResponseBody = mapper.readValue(entity.getContent(), errorResponseBodyType);
                throw ApiFacadeException.fromResponseBody(errorResponseBody);
            }
            EntityUtils.consume(entity);
            if (isError(responseBody)) {
                throw handleErrors(responseBody);
            } else if (statusCode < 200 || statusCode > 299) {
                throw ApiFacadeException.internalServerError("Request could not be completed successfully. Returned status code: " + statusCode);
            }
            return responseBody;
        } catch (IOException e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

    protected abstract JavaType getResponseBodyType(ObjectMapper mapper);

    protected abstract boolean isError(T responseBody);

    protected abstract ApiFacadeException handleErrors(T responseBody);

}
