/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.services;

import java.util.ArrayList;
import java.util.List;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.AllCapabilitiesConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ConfigConnector;
import com.openexchange.mobile.api.facade.connectors.responses.CapabilitiesResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ConfigResponseMto;
import com.openexchange.mobile.api.facade.models.Capability;
import com.openexchange.mobile.api.facade.models.Config;

import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class ConfigService {

    private final ConnectorFactoryFactory connectorFactoryFactory;

    public Config getConfig(SessionData sessionData) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        Config configuration = retrieveConfiguration(connectorFactory, sessionData);
        List<Capability> capabilities = retrieveCapabilities(connectorFactory, sessionData);
        boolean isMultipleAccountsEnabled = sessionData.getConfiguration().isReturnNonPrimaryAccounts() && userHasMultipleAccountsEnabled(capabilities);
        return configuration
                .withCapabilities(capabilities)
                .withMultipleAccountsEnabled(isMultipleAccountsEnabled);
    }

    public SessionData addConfiguration(SessionData sessionData) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        List<Capability> capabilities = retrieveCapabilities(connectorFactory, sessionData);
        return sessionData.withCapabilities(capabilities);
    }

    private Config retrieveConfiguration(ConnectorFactory connectorFactory, SessionData sessionData) {
        ConfigConnector connector = connectorFactory.getConfigConnector(sessionData);
        ConfigResponseMto response = connector.execute();
        return ConfigConnector.map(response);
    }

    private List<Capability> retrieveCapabilities(ConnectorFactory connectorFactory, SessionData sessionData) {
        AllCapabilitiesConnector connector = connectorFactory.getAllCapabilities(sessionData);
        CapabilitiesResponseMto response = connector.execute();
        List<Capability> remoteCapabilities = AllCapabilitiesConnector.map(response);
        List<Capability> capabilities = new ArrayList<>(remoteCapabilities.size() + 1);
        capabilities.addAll(remoteCapabilities);
        capabilities.add(new Capability("facade_accounts_sync"));
        capabilities.add(new Capability("facade_accounts_crud"));
        if (sessionData.getConfiguration().isReturnNonPrimaryAccounts() && userHasMultipleAccountsEnabled(capabilities)) {
            capabilities.add(new Capability("facade_multiple_accounts_enabled"));
        }
        return capabilities;
    }

    private boolean userHasMultipleAccountsEnabled(List<Capability> capabilities) {
        return capabilities.contains(new Capability("multiple_mail_accounts"));
    }

}
