/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import java.util.List;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.endpoints.requests.AccountRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.AutoconfigData;
import com.openexchange.mobile.api.facade.endpoints.requests.AutoconfigRequestBody;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountResponseBodyV2;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountsResponseBodyV2;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.services.AccountsService;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpStatus;

@Path("/api-facade/v2/accounts")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class AccountsEndpointV2 {

    private final AccountsService accountsService;

    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAccounts(@Context HttpServletRequest request) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Retrieve all accounts");
        String language = RequestUtil.getLanguage(request);
        List<MailAccountData> accounts = accountsService.getAccountsV2(sessionData, language);
        AccountsResponseBodyV2 responseBody = new AccountsResponseBodyV2(accounts);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

    @GET
    @Path("/{id}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAccount(@Context HttpServletRequest request, @PathParam("id") String accountId) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Retrieve specific account");
        String language = RequestUtil.getLanguage(request);
        MailAccountData account = accountsService.getAccountV2(sessionData, accountId, language);
        AccountResponseBodyV2 responseBody = new AccountResponseBodyV2(account);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response createAccount(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Create account");
        AccountRequestBody requestBody = JacksonUtil.parse(rawRequestBody, AccountRequestBody.class);
        boolean forceInsecureConnection = requestBody.getForceInsecureConnection() != null && requestBody.getForceInsecureConnection();
        boolean ignoreInvalidTransport = requestBody.getIgnoreInvalidTransport() != null && requestBody.getIgnoreInvalidTransport();
        MailAccountData account = accountsService.createAccount(sessionData, new MailAccountData(requestBody), forceInsecureConnection, ignoreInvalidTransport);
        AccountResponseBodyV2 responseBody = new AccountResponseBodyV2(account);
        return RequestUtil.jsonResponse(responseBody);
    }

    @PUT
    @Path("/{id}")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response updateAccount(@Context HttpServletRequest request, @PathParam("id") String accountId, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Update account");
        AccountRequestBody requestBody = JacksonUtil.parse(rawRequestBody, AccountRequestBody.class);
        if (!accountId.equals(requestBody.getId())) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.INVALID_ACCOUNT_DATA, "Account Ids don't match");
        }
        boolean forceInsecureConnection = requestBody.getForceInsecureConnection() != null && requestBody.getForceInsecureConnection();
        boolean ignoreInvalidTransport = requestBody.getIgnoreInvalidTransport() != null && requestBody.getIgnoreInvalidTransport();
        MailAccountData account = accountsService.updateAccount(sessionData, new MailAccountData(requestBody), forceInsecureConnection, ignoreInvalidTransport);
        AccountResponseBodyV2 responseBody = new AccountResponseBodyV2(account);
        return RequestUtil.jsonResponse(responseBody);
    }

    @DELETE
    @Path("/{id}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response deleteAccount(@Context HttpServletRequest request, @PathParam("id") String accountId) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Delete account");
        boolean isDeleted = accountsService.deleteAccount(sessionData, accountId);
        return Response.status(isDeleted ? HttpStatus.SC_OK : HttpStatus.SC_NOT_FOUND).build();
    }

    @POST
    @Path("/autoconfig")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response autoconfigAccount(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Autoconfigure account");
        AutoconfigRequestBody requestBody = JacksonUtil.parse(rawRequestBody, AutoconfigRequestBody.class);
        RequestUtil.checkParam("mailAddress", requestBody.getMailAddress());
        RequestUtil.checkParam("password", requestBody.getPassword());
        AutoconfigData autoconfigData = new AutoconfigData(requestBody);
        MailAccountData template = accountsService.autoconfigAccount(sessionData, autoconfigData);
        AccountResponseBodyV2 responseBody = new AccountResponseBodyV2(template);
        return RequestUtil.jsonResponse(responseBody);
    }

}
