/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.connectors.impl;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.BaseConnector;
import com.openexchange.mobile.api.facade.connectors.RequestConfiguration;
import com.openexchange.mobile.api.facade.connectors.responses.FoldersResponseMto;
import com.openexchange.mobile.api.facade.models.Folder;

import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;

public class AllFoldersConnector extends BaseConnector<FoldersResponseMto> {

    private static final int TYPE_DRAFTS = 9;

    private static final int TYPE_INBOX = 7;

    private static final int TYPE_SENT = 10;

    private static final int TYPE_SPAM = 11;

    private static final int TYPE_TRASH = 12;

    private static final int TYPE_PRIVATE = 1;

    private static final int TYPE_LEGACY = 7;

    private static final int TYPE_SHARED = 3;

    private static final int TYPE_PUBLIC = 2;

    private String rootFolderId;

    private String language;

    public AllFoldersConnector(RequestConfiguration configuration, SessionData sessionData, String rootFolderId, String language) {
        super(configuration, sessionData, "folders", FoldersResponseMto.class);
        this.rootFolderId = rootFolderId;
        this.language = language;
    }

    @Override
    protected HttpUriRequest createRequest() {
        RequestBuilder builder = RequestBuilder.post(createUrl());
        builder.setCharset(Charset.forName("UTF-8"));
        addSessionData(builder);
        builder.addParameter("action", "allVisible");
        builder.addParameter("content_type", "mail");
        builder.addParameter("columns", "1,20,300,302,316,306");
        if (!Strings.isEmpty(rootFolderId)) {
            builder.addParameter("root", rootFolderId);
        }
        if (!Strings.isEmpty(language)) {
            builder.addParameter("language", language);
        }
        return builder.build();
    }

    public static List<Folder> map(Object[][] data, String archiveFolderId, boolean includeNonPrivateFolders) {
        if (data == null) {
            return Collections.emptyList();
        }
        List<Folder> folders = new ArrayList<>();
        for (Object[] folderData : data) {
            Folder folder = AllFoldersConnector.map(folderData);
            if (folder != null && isMailFolder(folder, includeNonPrivateFolders)) {
                if (folder.getId().equals(archiveFolderId)) {
                    folder = folder.withType(Folder.Type.ARCHIVE);
                }
                folders.add(folder);
            }
        }
        return folders;
    }

    public static Folder map(Object[] data) {
        String folderId = (String) data[0];
        String parentFolderId = (String) data[1];
        String name = (String) data[2];
        Integer folderType = (Integer) data[3];
        Integer standardFolderType = (Integer) data[4];
        Folder.Type type = lookupType(standardFolderType, folderType);
        boolean isUserVisible = isUserVisible(data[5]);
        return isUserVisible ? new Folder(folderId, parentFolderId, name, type) : null;
    }

    private static Folder.Type lookupType(Integer standardFolderType, Integer folderType) {
        Folder.Type type = lookupByStandardFolderType(standardFolderType);
        if (Folder.Type.UNKNOWN.equals(type)) {
            type = lookupByAccessType(folderType);
        }
        return type;
    }

    private static Folder.Type lookupByStandardFolderType(Integer standardFolderType) {
        switch (standardFolderType) {
        case TYPE_DRAFTS:
            return Folder.Type.DRAFT;

        case TYPE_INBOX:
            return Folder.Type.INBOX;

        case TYPE_SENT:
            return Folder.Type.SENT;

        case TYPE_SPAM:
            return Folder.Type.SPAM;

        case TYPE_TRASH:
            return Folder.Type.TRASH;

        default:
            return Folder.Type.UNKNOWN;
        }
    }

    private static Folder.Type lookupByAccessType(Integer type) {
        switch (type) {
        case TYPE_PRIVATE:
        case TYPE_LEGACY:
            return Folder.Type.PRIVATE;

        case TYPE_SHARED:
            return Folder.Type.SHARED;

        case TYPE_PUBLIC:
            return Folder.Type.PUBLIC;

        default:
            return Folder.Type.PRIVATE;
        }
    }

    private static boolean isMailFolder(Folder folder, boolean includeNonPrivateFolders) {
        boolean isSubFolder = folder.getParentId() != null && folder.getId().startsWith(folder.getParentId());
        return isSubFolder && (includeNonPrivateFolders || (!Folder.Type.PUBLIC.equals(folder.getType()) && !Folder.Type.SHARED.equals(folder.getType())));
    }

    private static boolean isUserVisible(Object data) {
        if (data instanceof List) {
            List<?> permissionList = (List<?>) data;
            for (Object obj : permissionList) {
                if (obj instanceof Map) {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> permissions = (Map<String, Object>) obj;
                    Object bits = permissions.get("bits");
                    if (bits instanceof Integer) {
                        return ((Integer) bits).intValue() > 64;
                    }
                }
            }
        }
        return false;
    }

}
