/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.connectors.impl;

import java.util.ArrayList;
import java.util.List;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.BaseConnector;
import com.openexchange.mobile.api.facade.connectors.RequestConfiguration;
import com.openexchange.mobile.api.facade.connectors.responses.ArrayDataResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.AccountMto;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.utils.AccountUtil;

import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;

public class AccountsConnector extends BaseConnector<ArrayDataResponseMto> {

    public AccountsConnector(RequestConfiguration configuration, SessionData sessionData) {
        super(configuration, sessionData, "account", ArrayDataResponseMto.class);
    }

    @Override
    protected HttpUriRequest createRequest() {
        RequestBuilder builder = RequestBuilder.get(createUrl());
        addSessionData(builder);
        builder.addParameter("action", "all");
        builder.addParameter("columns", "1001,1002,1006,1007,1009,1010,1011,1012,1039,1042,1046,1037,1005,1017,1015,1016,1018,1044,1047,1032,1033,1034,1043,1021,1019,1020,1022,1045,1023,1048,1008,1030,1031,1025");
        return builder.build();
    }

    public static List<AccountMto> map(Object[][] data) {
        List<AccountMto> accounts = new ArrayList<>(data.length);
        for (int index = 0; index < data.length; index++) {
            accounts.add(map(data[index]));
        }
        return accounts;
    }

    public static AccountMto map(Object[][] data, String accountId) {
        for (int index = 0; index < data.length; index++) {
            if (accountId.equals(data[index][0].toString())) {
                return map(data[index]);
            }
        }
        throw ApiFacadeException.accountNotFound(accountId);
    }

    public static AccountMto map(Object[] data) {
        String id = Integer.toString((Integer) data[0]);
        String accountName = (String) data[2];
        String primaryAddress = (String) data[3];
        String trashFolderName = (String) data[4];
        String sentFolderName = (String) data[5];
        String draftsFolderName = (String) data[6];
        String spamFolderName = (String) data[7];
        List<String> aliases = AccountUtil.getAliases(primaryAddress, (String) data[8]);
        String archiveFolderId = (String) data[9];
        String rootFolderId = (String) data[10];
        String personal = (String) data[11];
        String transportUrl = (String) data[12];
        String mailProtocol = (String) data[13];
        boolean isPop3 = mailProtocol != null && mailProtocol.startsWith("pop3");
        return new AccountMto(id, accountName, primaryAddress, aliases, transportUrl, rootFolderId, trashFolderName, sentFolderName, draftsFolderName, spamFolderName, archiveFolderId, personal, isPop3);
    }

    public static List<MailAccountData> mapToMailAccountData(Object[][] data) {
        List<MailAccountData> accounts = new ArrayList<>(data.length);
        for (int index = 0; index < data.length; index++) {
            accounts.add(mapToMailAccountData(data[index]));
        }
        return accounts;
    }

    public static MailAccountData mapToMailAccountData(Object[] data) {
        String primaryAddress = (String) data[3];
        return new MailAccountData(
                Integer.toString((Integer) data[0]),
                (String) data[2],
                (String) data[11],
                (String) data[1],
                null,
                null,
                null,
                null,
                primaryAddress,
                AccountUtil.getAliases(primaryAddress, (String) data[8]),
                (String) data[13],     // 1017
                (String) data[14],     // 1015
                (Integer) data[15],    // 1016
                (Boolean) data[16],    // 1018
                (Boolean) data[17],    // 1044
                (Integer) data[18],    // 1047
                false,                 // TODO: Implement me.
                stringToInteger((String) data[19]),    // 1032
                (Boolean) data[20],    // 1033
                (Boolean) data[21],    // 1034
                (String) data[22],     // 1043
                (String) data[23],     // 1021
                (String) data[24],     // 1019
                (Integer) data[25],    // 1020
                (Boolean) data[26],    // 1022
                (Boolean) data[27],    // 1045
                (String) data[28],     // 1023
                (Integer) data[29],    // 1048
                false,                 // TODO: Implement me.
                (String) data[30],     // 1008
                (String) data[10],
                (String) data[4],
                (String) data[5],
                (String) data[6],
                (String) data[7],
                (String) data[31],     // 1030
                (String) data[32],     // 1031
                null,                  // TODO: Implement me.
                (String) data[9],
                (Boolean) data[33],     // 1025
                null,
                null,
                null
        );
    }

    private static Integer stringToInteger(String value) {
        if (value == null) {
            return null;
        }
        return Integer.valueOf(value);
    }

}
