/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.auth.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.Connector;
import com.openexchange.mobile.api.facade.connectors.RequestConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.CookieStore;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;

@Slf4j
public class LoginConnector implements Connector<SessionData, LoginResponseMto> {

    private final RequestConfiguration configuration;

    private final LoginCredentials credentials;

    public LoginConnector(RequestConfiguration configuration, LoginCredentials credentials) {
        this.configuration = configuration;
        this.credentials = credentials;
    }

    @Override
    public String createUrl() {
        return configuration.getBaseUrl() + "login";
    }

    @Override
    public SessionData execute() throws ApiFacadeException {
        CloseableHttpClient httpClient = configuration.getHttpClient();
        CookieStore cookieStore = configuration.getCookieStore();
        HttpUriRequest request = createRequest();
        try (CloseableHttpResponse response = httpClient.execute(request)) {
            LoginResponseMto responseBody = handleResponse(response);
            if (responseBody.isError()) {
                throw ApiFacadeException.fromResponseBody(responseBody);
            }
            if (Strings.isEmpty(responseBody.getSession())) {
                throw ApiFacadeException.internalServerError("No session received.");
            }
            if (cookieStore.getCookies().size() == 0) {
                throw ApiFacadeException.internalServerError("No cookies received.");
            }
            SessionData sessionData = new SessionData(responseBody.getSession(), cookieStore.getCookies());
            sessionData.setProperty(SessionData.PROPERTY_CONTEXT_ID, Long.toString(responseBody.getContextId()));
            sessionData.setProperty(SessionData.PROPERTY_USER_ID, Long.toString(responseBody.getUserId()));
            return sessionData;
        } catch (Exception e) {
            if (e instanceof ConnectTimeoutException) {
                throw ApiFacadeException.gatewayTimeoutError(e);
            }
            throw ApiFacadeException.internalServerError(e);
        }
    }

    private HttpUriRequest createRequest() {
        String clientId = "open-xchange-mobile-api-facade";
        if (!Strings.isEmpty(credentials.getClientId())) {
            clientId += "-" + credentials.getClientId();
        }
        RequestBuilder builder = RequestBuilder.post(createUrl());
        builder.setCharset(Charset.forName("UTF-8"));
        String hostname = credentials.getHost();
        String xHostname = credentials.getXHost();
        if (credentials.getHostConfiguration().isForwardHostHeader()) {
            if (!Strings.isEmpty(hostname)) {
                builder.addHeader(RequestUtil.HEADER_HOST, hostname);
            }
            if (!Strings.isEmpty(xHostname)) {
                builder.addHeader(RequestUtil.HEADER_X_HOST, xHostname);
            }
        }
        String forwardedFor = credentials.getForwardedFor();
        if (!Strings.isEmpty(forwardedFor)) {
            builder.addHeader(RequestUtil.HEADER_X_FORWARDED_FOR, forwardedFor);
        }
        String userAgent = credentials.getUserAgent();
        if (userAgent != null) {
            builder.addHeader(RequestUtil.HEADER_USER_AGENT, userAgent);
        }
        String trackingId = credentials.getTrackingId();
        if (trackingId != null) {
            builder.addParameter("trackingId", trackingId);
        }
        return builder
                .addParameter("action", "login")
                .addParameter("name", credentials.getUsername())
                .addParameter("password", credentials.getPassword())
                .addParameter("client", clientId)
                .build();
    }

    @Override
    public LoginResponseMto handleResponse(HttpResponse response) {
        try {
            StatusLine statusLine = response.getStatusLine();
            int statusCode = statusLine.getStatusCode();
            HttpEntity entity = response.getEntity();
            if (log.isDebugEnabled()) {
                entity = new BufferedHttpEntity(response.getEntity());
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                entity.writeTo(baos);
                log.debug(new String(baos.toByteArray(), "UTF-8"));
            }
            if (statusCode == 200) {
                ObjectMapper mapper = JacksonUtil.getMapper();
                LoginResponseMto responseBody = mapper.readValue(entity.getContent(), LoginResponseMto.class);
                EntityUtils.consume(entity);
                return responseBody;
            }
            throw ApiFacadeException.internalServerError("Unable to log in");
        } catch (IOException e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

}
