/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.endpoints.responses.ctos;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.openexchange.mobile.api.facade.models.Account;
import com.openexchange.mobile.api.facade.models.Folder;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.Data;
import lombok.NonNull;

@Data
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AccountCto {

    @NonNull
    private final String id;

    @NonNull
    private final String accountName;

    private final String userName;

    private final String firstName;

    private final String lastName;

    @NonNull
    private final String primaryAddress;

    private final List<AliasCto> aliases;

    private final boolean canSend;

    @NonNull
    private final List<FolderCto> folders;

    @NonNull
    private final StandardFoldersCto standardFolders;

    private static final MapFunction<String, AliasCto> aliasesMapFunction = new MapFunction<String, AliasCto>() {
        @Override
        public AliasCto map(String address) {
            return new AliasCto(address);
        }
    };

    public AccountCto(Account account) {
        id = account.getId();
        accountName = account.getAccountName();
        userName = account.getUserName();
        firstName = account.getFirstName();
        lastName = account.getLastName();
        primaryAddress = account.getPrimaryAddress();
        aliases = ListUtil.map(account.getAliases(), aliasesMapFunction);
        canSend = account.isCanSend();
        if (account.getFolders() != null) {
            /* Java8: Use with Java 8.
            folders = account.getFolders().stream()
                    .map(folder -> new FolderCto(folder))
                    .collect(Collectors.toList());
            */
            folders = new ArrayList<>();
            for (Folder folder : account.getFolders()) {
                folders.add(new FolderCto(folder));
            }
            standardFolders = new StandardFoldersCto(account.getStandardFolders());
        } else {
            folders = null;
            standardFolders = null;
        }
    }

    public AccountCto(MailAccountData account) {
        id = account.getId();
        accountName = account.getAccountName();
        firstName = "";
        lastName = "";
        userName = "";
        primaryAddress = account.getPrimaryAddress();
        aliases = ListUtil.map(account.getAliases(), aliasesMapFunction);
        canSend = true;
        folders = null;
        standardFolders = null;
    }

}
