/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.openexchange.mobile.api.facade.endpoints.responses.ContactsResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.ctos.ContactCto;
import com.openexchange.mobile.api.facade.models.Contact;
import com.openexchange.mobile.api.facade.services.ContactsService;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.RequiredArgsConstructor;
import lombok.val;
import lombok.extern.slf4j.Slf4j;

@Path("/api-facade/v1/contacts")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class ContactsEndpoint {

    private final ContactsService contactsService;

    private static final MapFunction<Contact, ContactCto> contactMapFunction = new MapFunction<Contact, ContactCto>() {
        @Override
        public ContactCto map(Contact contact) {
            return new ContactCto(contact);
        }
    };

    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAllContacts(@Context HttpServletRequest request) {
        val sessionData = RequestUtil.getSessionData(request);
        log.info("Retrieving contacts");
        val contacts = contactsService.getAllContacts(sessionData);
        val contactCtos = ListUtil.map(contacts, contactMapFunction);
        val responseBody = new ContactsResponseBody(contactCtos);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

}
