/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.connectors.impl;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.BaseConnector;
import com.openexchange.mobile.api.facade.connectors.RequestConfiguration;
import com.openexchange.mobile.api.facade.connectors.responses.DeleteMailsResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.FolderMailMto;
import com.openexchange.mobile.api.facade.models.DeletedMails;

import lombok.SneakyThrows;

import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;

public class DeleteMailsConnector extends BaseConnector<DeleteMailsResponseMto> {

    private final String folderId;

    private final List<String> mailIds;

    private final boolean purge;

    public DeleteMailsConnector(RequestConfiguration configuration, SessionData sessionData, String folderId, List<String> mailIds, boolean purge) {
        super(configuration, sessionData, "mail", DeleteMailsResponseMto.class);
        this.folderId = folderId;
        this.mailIds = mailIds;
        this.purge = purge;
    }

    @SneakyThrows
    @Override
    protected HttpUriRequest createRequest() {
        RequestBuilder builder = RequestBuilder.put(createUrl());
        builder.setCharset(Charset.forName("UTF-8"));
        addSessionData(builder);
        builder.addParameter("action", "trash");
        builder.addParameter("folder", folderId);
        builder.addParameter("harddelete", Boolean.toString(purge));
        builder.setEntity(new StringEntity(createRequestBody(), "UTF-8"));
        return builder.build();
    }

    private String createRequestBody() {
        return "[\"" + Strings.join(mailIds, "\",\"") + "\"]";
    }

    public static DeletedMails map(DeleteMailsResponseMto response) {
        List<FolderMailMto> data = response != null ? response.getData() : null;
        if (data == null || data.size() == 0) {
            return new DeletedMails();
        }
        String trashFolderId = null;
        List<String> mailIds = new ArrayList<>();
        for (FolderMailMto folderMailMto : data) {
            if (folderMailMto == null) {
                mailIds.add(null);
                continue;
            }
            if (trashFolderId == null) {
                trashFolderId = folderMailMto.getFolderId();
            }
            mailIds.add(folderMailMto.getMailId());
        }
        return new DeletedMails(trashFolderId, mailIds);
    }

}
