/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.services;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.AllFoldersConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ClearFolderConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ConfigConnector;
import com.openexchange.mobile.api.facade.connectors.impl.CreateFolderConnector;
import com.openexchange.mobile.api.facade.connectors.impl.DeleteFolderConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ExamineMultipleFoldersConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ExpungeFolderConnector;
import com.openexchange.mobile.api.facade.connectors.impl.UpdateFolderConnector;
import com.openexchange.mobile.api.facade.connectors.responses.ConfigResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ExamineResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.FoldersResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.StringResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.ExamineResultMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.FoldersMto;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.models.Account;
import com.openexchange.mobile.api.facade.models.Config;
import com.openexchange.mobile.api.facade.models.CreatedFolder;
import com.openexchange.mobile.api.facade.models.ExamineResult;
import com.openexchange.mobile.api.facade.models.Folder;
import com.openexchange.mobile.api.facade.models.Folder.Type;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.models.UpdatedFolder;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

@RequiredArgsConstructor
@Slf4j
public class FoldersService {

    private static final HashSet<String> ERROR_CODES_AUTHENTICATION = new HashSet<String>();

    private final ConnectorFactoryFactory connectorFactoryFactory;

    @Setter
    private AccountsService accountsService;

    static {
        ERROR_CODES_AUTHENTICATION.add("MSG-0091");
        ERROR_CODES_AUTHENTICATION.add("MSG-0114");
        ERROR_CODES_AUTHENTICATION.add("MSG-1000");
        ERROR_CODES_AUTHENTICATION.add("MSG-1001");
        ERROR_CODES_AUTHENTICATION.add("MSG-1031");
        ERROR_CODES_AUTHENTICATION.add("OAUTH-0013");
        ERROR_CODES_AUTHENTICATION.add("OAUTH-0042");
        ERROR_CODES_AUTHENTICATION.add("OAUTH-0043");
        ERROR_CODES_AUTHENTICATION.add("OAUTH-0044");
        ERROR_CODES_AUTHENTICATION.add("pop3-2071");
    }

    public CreatedFolder createFolder(SessionData sessionData, String parentFolderId, String name) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        CreateFolderConnector connector = connectorFactory.getCreateFolderConnector(sessionData, parentFolderId, name);
        StringResponseMto response = connector.execute();
        String folderId = CreateFolderConnector.map(response);
        return new CreatedFolder(folderId, name);
    }

    public UpdatedFolder updateFolder(SessionData sessionData, String folderId, String newParentFolderId, String newName) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        UpdateFolderConnector connector = connectorFactory.getUpdateFolderConnector(sessionData, folderId, newParentFolderId, newName);
        StringResponseMto response = connector.execute();
        String newFolderId = UpdateFolderConnector.map(response);
        return new UpdatedFolder(newFolderId);
    }

    public void deleteFolder(SessionData sessionData, String folderId) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        DeleteFolderConnector connector = connectorFactory.getDeleteFolderConnector(sessionData, folderId);
        connector.execute();
    }

    public void clearFolder(SessionData sessionData, String folderId) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        ClearFolderConnector connector = connectorFactory.getClearFolderConnector(sessionData, folderId);
        connector.execute();
    }

    public void expungeFolder(SessionData sessionData, String folderId) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        ExpungeFolderConnector connector = connectorFactory.getExpungeFolderConnector(sessionData, folderId);
        connector.execute();
    }

    public List<Account> getAllMailFolders(final SessionData sessionData, final String language) {
        final ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        Config config = retrieveConfiguration(connectorFactory, sessionData);
        final boolean isNamespaceSet = "INBOX/".equals(config.getNamespace());
        List<Account> accounts = accountsService.getAccounts(connectorFactory, sessionData);
        return ListUtil.map(accounts, new MapFunction<Account, Account>() {

            @Override
            public Account map(Account account) {
                return addFoldersImpl(connectorFactory, sessionData, account, language, isNamespaceSet);
            }

        });
    }

    public Account addFolders(ConnectorFactory connectorFactory, SessionData sessionData, Account account, String language) {
        Config config = retrieveConfiguration(connectorFactory, sessionData);
        boolean isNamespaceSet = "INBOX/".equals(config.getNamespace());
        return addFoldersImpl(connectorFactory, sessionData, account, language, isNamespaceSet);
    }

    private Account addFoldersImpl(ConnectorFactory connectorFactory, SessionData sessionData, Account account, String language, boolean isNamespaceSet) {
        List<Folder> folders = retrieveFolders(connectorFactory, sessionData, account.getRootFolderId(), account.getArchiveFolderId(), language);
        if (!account.isPop3Account()) {
            examineFolders(connectorFactory, sessionData, folders);
        }
        account = account.withFolders(folders);
        adjustFolderParents(account, account.isPrimaryAccount() && isNamespaceSet);
        return account;
    }

    public MailAccountData addFolders(ConnectorFactory connectorFactory, SessionData sessionData, MailAccountData account, String language) {
        Config config = retrieveConfiguration(connectorFactory, sessionData);
        boolean isNamespaceSet = "INBOX/".equals(config.getNamespace());
        return addFoldersImpl(connectorFactory, sessionData, account, language, isNamespaceSet);
    }

    private MailAccountData addFoldersImpl(ConnectorFactory connectorFactory, SessionData sessionData, MailAccountData account, String language, boolean isNamespaceSet) {
        List<Folder> folders = retrieveFolders(connectorFactory, sessionData, account.getRootFolderId(), account.getArchiveFolderId(), language);
        if (!account.isPop3Account()) {
            examineFolders(connectorFactory, sessionData, folders);
        } else {
            examineResetFolders(folders);
        }
        account = account.withFolders(folders);
        adjustFolderParents(account, account.isPrimaryAccount() && isNamespaceSet);
        return account;
    }

    private Config retrieveConfiguration(ConnectorFactory connectorFactory, SessionData sessionData) {
        ConfigConnector configConnector = connectorFactory.getConfigConnector(sessionData);
        ConfigResponseMto response = configConnector.execute();
        return ConfigConnector.map(response);
    }

    private List<Folder> retrieveFolders(ConnectorFactory connectorFactory, SessionData sessionData, String rootFolderId, String archiveFolderId, String language) {
        boolean includeNonPrivateFolders = sessionData.getConfiguration().isIncludeNonPrivateFolders();
        AllFoldersConnector connector = connectorFactory.getAllFoldersConnector(sessionData, rootFolderId, language);
        FoldersMto data;
        try {
            FoldersResponseMto response = connector.execute();
            data = response.getData();
        } catch (ApiFacadeException e) {
            String errorCode = e.getErrorCode();
            log.info("Retrieve folders failed with code: " + errorCode);
            // OMFACADE-633: Throw authentication errors directly back to the clients.
            if (ERROR_CODES_AUTHENTICATION.contains(errorCode)) {
                throw e;
            }
            // Return fake INBOX folder to not crash older iOS clients.
            Folder inboxFolder = new Folder(rootFolderId + "/INBOX", rootFolderId, "INBOX", Type.INBOX);
            return Collections.singletonList(inboxFolder);
        }
        return data != null
                ? AllFoldersConnector.map(data.getPrivateFolders(), archiveFolderId, includeNonPrivateFolders)
                : Collections.<Folder>emptyList();
    }

    /*private*/ void examineFolders(ConnectorFactory connectorFactory, SessionData sessionData, List<Folder> folders) {
        ExamineMultipleFoldersConnector connector = connectorFactory.getExamineMultipleFoldersConnector(sessionData, folders);
        List<ExamineResponseMto> responses = connector.execute();

        for (int folderIndex = 0, responseIndex = 0; folderIndex < folders.size() && responseIndex < responses.size(); folderIndex++, responseIndex++) {
            Folder folder = folders.get(folderIndex);
            ExamineResponseMto response = responses.get(responseIndex);
            ExamineResultMto result = response.getData();
            if (!response.isError() && result != null) {
                folder.setValidity(result.getValidity());
                folder.setNext(result.getNext());
                folder.setTotal(result.getTotal());
                folder.setUnread(result.getUnread());
            } else {
                folders.remove(folderIndex);
                folderIndex--;
            }
        }
    }

    public ExamineResult examineFolder(SessionData sessionData, String folderId, String validity, String next) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        ExamineMultipleFoldersConnector connector = connectorFactory.getExamineFolderConnector(sessionData, folderId);
        List<ExamineResponseMto> responses = connector.execute();
        ExamineResultMto response = responses.get(0).getData();
        String newValidity = response.getValidity();
        String newNext = response.getNext();
        boolean changed = !validity.equals(newValidity) || !next.equals(newNext);
        return new ExamineResult(folderId, changed, newValidity, newNext);
    }

    private void examineResetFolders(List<Folder> folders) {
        for (int index = 0; index < folders.size(); index++) {
            Folder folder = folders.get(index);
            folder.setValidity(null);
            folder.setNext(null);
            folder.setTotal(-1);
            folder.setUnread(-1);
        }
    }

    private void adjustFolderParents(Account account, boolean isNamespaceSet) {
        if (!isNamespaceSet) {
            return;
        }
        Folder inboxFolder = account.getStandardFolders().getInboxFolder();
        if (inboxFolder != null) {
            String inboxFolderId = inboxFolder.getId();
            String parentFolderId = inboxFolder.getParentId();
            for (Folder folder : account.getFolders()) {
                if (inboxFolderId.equals(folder.getParentId())) {
                    folder.setParentId(parentFolderId);
                }
            }
        }
    }

    private void adjustFolderParents(MailAccountData account, boolean isNamespaceSet) {
        if (!isNamespaceSet) {
            return;
        }
        Folder inboxFolder = account.getStandardFolders().getInboxFolder();
        if (inboxFolder != null) {
            String inboxFolderId = inboxFolder.getId();
            String parentFolderId = inboxFolder.getParentId();
            for (Folder folder : account.getFolders()) {
                if (inboxFolderId.equals(folder.getParentId())) {
                    folder.setParentId(parentFolderId);
                }
            }
        }
    }

}
