/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;

import lombok.experimental.UtilityClass;

@UtilityClass
public class WeakforceUtil {

    public String generatePasswordHash(HostConfiguration configuration, LoginCredentials credentials) {
        StringBuilder s = new StringBuilder(32)
                .append(configuration.getWeakforceHashNonce())
                .append(credentials.getUsername())
                .append('\0')
                .append(credentials.getPassword());
        byte[] digest = getDigest(s.toString(), configuration.getWeakforceHashMechanism());
        return toHexString(((digest[0] & 0xFF) << 4) | ((digest[1] & 0xFF) >> 4), 4);
    }

    /**
     * All possible chars for representing a number as a String
     */
    private final char[] digits = {
            '0', '1', '2', '3', '4', '5',
            '6', '7', '8', '9', 'a', 'b',
            'c', 'd', 'e', 'f', 'g', 'h',
            'i', 'j', 'k', 'l', 'm', 'n',
            'o', 'p', 'q', 'r', 's', 't',
            'u', 'v', 'w', 'x', 'y', 'z'
    };

    /**
     * Converts the two-byte integer to a HEX string.
     */
    private String toHexString(int i, int padding) {
        char[] buf = new char[16];
        int charPos = 16;
        int mask = (1 << 4) - 1;

        do {
            buf[--charPos] = digits[i & mask];
            i >>>= 4;
        } while (i != 0);

        while ((16 - charPos) < padding) {
            buf[--charPos] = '0';
        }

        return new String(buf, charPos, 16 - charPos);
    }

    /**
     * Gets the digest of specified string.
     *
     * @param string The string to hash
     * @param algorithm The name of the algorithm
     * @return The digest
     */
    private byte[] getDigest(String string, String algorithm) {
        try {
            MessageDigest md = MessageDigest.getInstance(algorithm);
            md.update(string.getBytes("UTF-8"));
            return md.digest();
        } catch (NoSuchAlgorithmException e) {
            throw ApiFacadeException.weakforceInvalidHashAlgorithm(algorithm);
        } catch (UnsupportedEncodingException e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

}
