/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.utils;

import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.VersionConnector;
import com.openexchange.mobile.api.facade.connectors.responses.VersionResponseMto;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

@UtilityClass
@Slf4j
public class VersionCheckerUtil {

    static final String SUPPORTED_VERSION_NUMBER = "7.10.5";

    static final int SUPPORTED_MINIMUM_REVISION = 0;

    static final long MILLIS_SLEEP_TIME = 10_000;

    public void check(ConnectorFactoryFactory connectorFactoryFactory) {
        String remoteVersion = retrieveRemoteVersion(connectorFactoryFactory, MILLIS_SLEEP_TIME);
        checkVersion(remoteVersion);
    }

    private String retrieveRemoteVersion(ConnectorFactoryFactory connectorFactoryFactory, long millisSleep) {
        String version = null;
        while (version == null) {
            try {
                version = retrieveRemoteVersion(connectorFactoryFactory);
                log.info("Remote middleware version: " + version);
            } catch (Exception e) {
                log.warn("Failed to retrieve remote middleware version. Waiting for next try");
                try {
                    Thread.sleep(millisSleep);
                } catch (InterruptedException e2) {
                    e2.printStackTrace();
                }
            }
        }
        return version;
    }

    private String retrieveRemoteVersion(ConnectorFactoryFactory connectorFactoryFactory) {
        try {
            ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
            VersionConnector connector = connectorFactory.getVersionConnector();
            VersionResponseMto response = connector.execute();
            return response.getData().getVersion();
        } catch (Exception e) {
            throw new RuntimeException("Failed to retrieve middleware version: " + e.getMessage());
        }
    }

    void checkVersion(String remoteVersion) {
        if (remoteVersion == null) {
            throw new RuntimeException("Remote version is null");
        }
        String message = String.format("Remote version is incompatible: %s. This version of the facade is compatible with version %s-Rev%d or newer", remoteVersion, SUPPORTED_VERSION_NUMBER, SUPPORTED_MINIMUM_REVISION);
        if (!remoteVersion.startsWith(SUPPORTED_VERSION_NUMBER)) {
            throw new RuntimeException(message);
        }
        try {
            String revisionString = remoteVersion.substring(SUPPORTED_VERSION_NUMBER.length() + 4);
            int revision = Integer.parseInt(revisionString);
            if (revision < SUPPORTED_MINIMUM_REVISION) {
                throw new RuntimeException(message);
            }
        } catch (NumberFormatException e) {
            throw new RuntimeException("Remote version is parseable");
        }
    }

}
