/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.openexchange.mobile.api.facade.models.MailContact;

import lombok.experimental.UtilityClass;

@UtilityClass
public class MailContactUtil {

    private static final Pattern PATTERN_BEGINS_WITH_QUOTE = Pattern.compile("^[\\\"'\\\\\\s]");

    private static final Pattern PATTERN_REORDER_DISPLAY_NAME = Pattern.compile("^([^,.\\(\\)]+),\\s([^,.\\(\\)]+)$");

    public List<MailContact> convert(List<List<String>> rawList) {
        if (ListUtil.isEmpty(rawList)) {
            return null;
        }

        List<MailContact> mailContacts = new ArrayList<>(rawList.size() / 2);

        for (List<String> entryList : rawList) {
            if (entryList.size() != 2) {
                throw new IllegalArgumentException("Entry list has not 2 elements");
            }

            String name = sanitizeName(entryList.get(0));
            String address = entryList.get(1);
            if (name != null) {
                name = reorderDisplayName(name);
            }
            if (address != null && address.contains("@")) {
                MailContact mailContact = new MailContact(name, address);
                mailContacts.add(mailContact);
            }
        }

        return mailContacts;
    }

    public String sanitizeName(String name) {
        if (name != null) {
            // Remove outer quotes.
            while (name.length() > 1
                    && PATTERN_BEGINS_WITH_QUOTE.matcher(name).lookingAt()
                    && name.charAt(0) == name.charAt(name.length() - 1)) {
                name = name.substring(1, name.length() - 1).trim();
            }

            // Unescape inner quotes.
            name = name.replaceAll("\\\"", "\"");

            // Unescape escaped backslashes.
            name = name.replaceAll("\\{2}", "\\");

            if (name.length() == 0) {
                name = null;
            }
        }
        return name;
    }

    public String reorderDisplayName(String name) {
        Matcher matcher = PATTERN_REORDER_DISPLAY_NAME.matcher(name);
        if (matcher.matches()) {
            name = matcher.group(2) + " " + matcher.group(1);
        }
        return name;
    }

}
