/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.osgi;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.AuthenticationService;
import com.openexchange.mobile.api.facade.auth.impl.DefaultAuthenticationService;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.weakforce.WeakforceService;
import com.openexchange.mobile.api.facade.weakforce.WeakforceWrapperAuthenticationService;
import com.openexchange.osgi.SimpleRegistryListener;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.osgi.framework.ServiceReference;

@RequiredArgsConstructor
@Slf4j
public class MultipleAuthenticationServicesHandler {

    private final class AuthenticationServiceRegistryListener implements SimpleRegistryListener<AuthenticationService> {
        @Override
        public void added(ServiceReference<AuthenticationService> serviceReference, AuthenticationService service) {
            String name = service.getClass().getName();
            authenticationServices.put(name, service);
            log.debug("Registered authentication service: " + service.getClass().getName());
        }

        @Override
        public void removed(ServiceReference<AuthenticationService> serviceReference, AuthenticationService service) {
            String name = service.getClass().getName();
            authenticationServices.remove(name);
            log.debug("Unregistered authentication service: " + service.getClass().getName());
        }
    }

    private final WeakforceService weakforceService;

    /*private*/ final Map<String, AuthenticationService> authenticationServices = new ConcurrentHashMap<>();

    @Getter
    private final SimpleRegistryListener<AuthenticationService> serviceListener = new AuthenticationServiceRegistryListener();

    public AuthenticationService getAuthenticationService(HostConfiguration configuration) {
        log.debug("Retrieving authentication service");
        String name = configuration.getAuthenticationClassName();
        if (Strings.isEmpty(name)) {
            name = DefaultAuthenticationService.class.getName();
        }
        AuthenticationService service = authenticationServices.get(name);
        if (service == null) {
            throw ApiFacadeException.authenticationNotFound();
        }
        log.debug("Using authentication service " + service.getClass().getName());
        if (configuration.isWeakforceEnabled()) {
            service = new WeakforceWrapperAuthenticationService(configuration, weakforceService, service);
        }
        return service;
    }

}
