/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import java.util.List;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountIdsResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountsResponseBody;
import com.openexchange.mobile.api.facade.models.Account;
import com.openexchange.mobile.api.facade.services.AccountsService;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Path("/api-facade/v1/accounts")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class AccountsEndpoint {

    private final AccountsService accountsService;

    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAccounts(@Context HttpServletRequest request) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Retrieve all accounts");
        List<Account> accounts = accountsService.getAccounts(sessionData);
        AccountsResponseBody responseBody = new AccountsResponseBody(accounts);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

    @GET
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/ids")
    public Response getAccountList(@Context HttpServletRequest request) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Retrieve account list");
        List<String> accounts = accountsService.getAccountList(sessionData);
        AccountIdsResponseBody responseBody = new AccountIdsResponseBody(accounts);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

    @GET
    @Path("/{id}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAccount(@Context HttpServletRequest request, @PathParam("id") String accountId) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Retrieve specific account");
        String language = RequestUtil.getLanguage(request);
        Account account = accountsService.getAccount(sessionData, accountId, language);
        AccountResponseBody responseBody = new AccountResponseBody(account);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

}
