/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.weakforce;

import com.openexchange.mobile.api.facade.auth.AuthenticationService;
import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.endpoints.requests.LogoutRequestBody;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.weakforce.models.AllowResponse;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@RequiredArgsConstructor
@Slf4j
public class WeakforceWrapperAuthenticationService implements AuthenticationService {

    private final HostConfiguration configuration;

    private final WeakforceService weakforceService;

    private final AuthenticationService wrappedService;

    @Override
    public SessionData authenticate(LoginCredentials credentials) {
        boolean loginSuccessful = false;
        boolean policyReject = false;
        try {
            AllowResponse response = callWeakforceAllow(credentials);
            policyReject = response != null && response.isRejected();
            if (policyReject) {
                throw ApiFacadeException.invalidCredentials();
            }
            SessionData sessionData = wrappedService.authenticate(credentials);
            loginSuccessful = true;
            return sessionData;
        } catch (ApiFacadeException e) {
            loginSuccessful = false;
            throw e;
        } finally {
            callWeakforceReport(credentials, loginSuccessful, policyReject);
        }
    }

    private AllowResponse callWeakforceAllow(LoginCredentials credentials) {
        AllowResponse allowResponse = null;
        try {
            allowResponse = weakforceService.allow(configuration, credentials);
            if (allowResponse.isDelayed()) {
                sleep(allowResponse.getSecondsToDelay());
            }
        } catch (ApiFacadeException e) {
            log.error("Failed to call weakforce.allow");
        }
        return allowResponse;
    }

    private void callWeakforceReport(LoginCredentials credentials, boolean loginSuccessful, boolean policyReject) {
        try {
            weakforceService.report(configuration, credentials, loginSuccessful, policyReject);
        } catch (ApiFacadeException e) {
            log.error("Failed to call weakforce.report");
        }
    }

    @Override
    public void logout(SessionData sessionData, LogoutRequestBody requestBody) {
        wrappedService.logout(sessionData, requestBody);
    }

    private void sleep(int secondsToSleep) {
        try {
            log.info(String.format("Sleeping for %d seconds", secondsToSleep));
            Thread.sleep(secondsToSleep * 1000);
        } catch (InterruptedException e) {
            log.error(String.format("Failed to sleep for %d seconds", secondsToSleep), e);
        }
    }

}
