/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ws.rs.core.NewCookie;
import javax.ws.rs.core.Response;

import lombok.experimental.UtilityClass;

import org.apache.http.cookie.Cookie;
import org.apache.http.impl.cookie.BasicClientCookie;

@UtilityClass
public class CookieUtil {

    private static final int SECONDS_TWO_WEEKS = 1_209_600;

    private static final int MILLIS_ONE_WEEK = 604_800_000;

    public void addToResponse(Response.ResponseBuilder builder, List<Cookie> cookies) {
        /* Java8: Use this with Java 8.
        convertForEndpoint(cookies)
                .forEach(cookie -> builder.cookie(cookie));
        */
        for (NewCookie cookie : convertForEndpoint(cookies)) {
            builder.cookie(cookie);
        }
    }

    public List<NewCookie> convertForEndpoint(List<Cookie> cookies) {
        /* Java8: Use this with Java 8.
        return cookies.stream()
                .map(cookie -> convertForEndpoint(cookie))
                .collect(toList());
        */
        List<NewCookie> result = new ArrayList<>();
        for (Cookie cookie : cookies) {
            NewCookie newCookie = convertForEndpoint(cookie);
            result.add(newCookie);
        }
        return result;
    }

    public NewCookie convertForEndpoint(Cookie cookie) {
        return new NewCookie(cookie.getName(), cookie.getValue(), cookie.getPath(), null, null, calculateMaxAge(cookie), cookie.isSecure());
    }

    private int calculateMaxAge(Cookie cookie) {
        int maxAge;
        Date expiresDate = cookie.getExpiryDate();
        if (expiresDate != null) {
            maxAge = (int) ((expiresDate.getTime() - System.currentTimeMillis()) / 1000);
        } else {
            maxAge = SECONDS_TWO_WEEKS;
        }
        return maxAge;
    }

    public List<Cookie> convertForConnector(javax.servlet.http.Cookie[] cookies, String domain) {
        List<Cookie> convertedCookies = new ArrayList<>();
        if (cookies != null) {
            for (javax.servlet.http.Cookie requestCookie : cookies) {
                BasicClientCookie cookie = new BasicClientCookie(requestCookie.getName(), requestCookie.getValue());
                cookie.setDomain(domain);
                cookie.setPath("/");
                cookie.setSecure(requestCookie.getSecure());
                cookie.setExpiryDate(new Date(System.currentTimeMillis() + MILLIS_ONE_WEEK));
                convertedCookies.add(cookie);
            }
        }
        return convertedCookies;
    }

}
