/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import java.util.List;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.endpoints.requests.ExamineFolderRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.FolderCreateRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.FolderRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.FolderUpdateRequestBody;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountsResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.CreateFolderResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.ExamineFolderResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.UpdatedFolderResponseBody;
import com.openexchange.mobile.api.facade.models.Account;
import com.openexchange.mobile.api.facade.models.CreatedFolder;
import com.openexchange.mobile.api.facade.models.ExamineResult;
import com.openexchange.mobile.api.facade.models.UpdatedFolder;
import com.openexchange.mobile.api.facade.services.FoldersService;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Path("/api-facade/v1/folders")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class FoldersEndpoint {

    private static final String FIELD_FOLDER_ID = "folderId";

    private final FoldersService foldersService;

    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response createFolder(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Create folder");
        FolderCreateRequestBody requestBody = JacksonUtil.parse(rawRequestBody, FolderCreateRequestBody.class);
        CreatedFolder folder = foldersService.createFolder(sessionData, requestBody.getParentFolderId(), requestBody.getName());
        CreateFolderResponseBody responseBody = new CreateFolderResponseBody(folder);
        return RequestUtil.jsonResponse(201, responseBody);
    }

    @PUT
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response updateFolder(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Update folder");
        FolderUpdateRequestBody requestBody = JacksonUtil.parse(rawRequestBody, FolderUpdateRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        UpdatedFolder folder = foldersService.updateFolder(sessionData, requestBody.getFolderId(), requestBody.getNewParentFolderId(), requestBody.getNewName());
        UpdatedFolderResponseBody responseBody = new UpdatedFolderResponseBody(folder);
        return RequestUtil.jsonResponse(200, responseBody);
    }

    @DELETE
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response deleteFolder(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Delete folder");
        FolderRequestBody requestBody = JacksonUtil.parse(rawRequestBody, FolderRequestBody.class);
        foldersService.deleteFolder(sessionData, requestBody.getFolderId());
        return RequestUtil.emptyJsonResponse();
    }

    @DELETE
    @Path("/clear")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response clearFolder(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Clear folder");
        FolderRequestBody requestBody = JacksonUtil.parse(rawRequestBody, FolderRequestBody.class);
        foldersService.clearFolder(sessionData, requestBody.getFolderId());
        return RequestUtil.emptyJsonResponse();
    }

    @DELETE
    @Path("/expunge")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response expungeFolder(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Expunge folder");
        FolderRequestBody requestBody = JacksonUtil.parse(rawRequestBody, FolderRequestBody.class);
        foldersService.expungeFolder(sessionData, requestBody.getFolderId());
        return RequestUtil.emptyJsonResponse();
    }

    @GET
    @Path("/mail")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAllMailFolders(@Context HttpServletRequest request) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Retrieve mail folders");
        String language = RequestUtil.getLanguage(request);
        List<Account> accounts = foldersService.getAllMailFolders(sessionData, language);
        AccountsResponseBody accountsResponseBody = new AccountsResponseBody(accounts);
        return RequestUtil.createCachableResponse(request, accountsResponseBody);
    }

    @POST
    @Path("/mail/checknew")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response examineFolder(@Context HttpServletRequest request, String rawRequestBody) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        log.info("Examine folder");
        ExamineFolderRequestBody requestBody = JacksonUtil.parse(rawRequestBody, ExamineFolderRequestBody.class);
        ExamineResult result = foldersService.examineFolder(sessionData, requestBody.getFolderId(), requestBody.getValidity(), requestBody.getNext());
        ExamineFolderResponseBody responseBody = new ExamineFolderResponseBody(result);
        return RequestUtil.jsonResponse(result.isChanged() ? 200 : 304, responseBody);
    }
}
