/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.endpoints.responses.ctos;

import java.util.Iterator;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.models.Attachment;
import com.openexchange.mobile.api.facade.models.Mail;
import com.openexchange.mobile.api.facade.models.MailContact;
import com.openexchange.mobile.api.facade.models.Priority;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.Data;

@Data
@JsonInclude(JsonInclude.Include.NON_NULL)
public class MailCto {

    private static final MapFunction<MailContact, MailContactCto> contactsMapFunction = new MapFunction<MailContact, MailContactCto>() {
        @Override
        public MailContactCto map(MailContact mailContact) {
            return new MailContactCto(mailContact);
        }
    };

    private static final MapFunction<Attachment, AttachmentCto> partMapFunction = new MapFunction<Attachment, AttachmentCto>() {
        @Override
        public AttachmentCto map(Attachment attachment) {
            return new AttachmentCto(attachment);
        }
    };

    private final String id;

    private final String rootMailId;

    private final String folderId;

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    private final MailFlagsCto flags;

    private final Integer color;

    private final String priority;

    private final Long receivedDate;

    private final Long sentDate;

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    private final List<MailContactCto> from;

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    private final List<MailContactCto> to;

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    private final List<MailContactCto> cc;

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    private final List<MailContactCto> bcc;

    private final List<MailContactCto> replyTo;

    private final String messageId;

    private final String subject;

    private final String teaser;

    private final String contentType;

    private final String bodyType;

    private final String body;

    private final Long size;

    // Deprecated. Please use 'hasAttachment'.
    private final Boolean realAttachment;

    private final Boolean hasAttachment;

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    private final List<AttachmentCto> attachments;

    public MailCto(Mail mail) {
        id = mail.getId();
        rootMailId = mail.getRootMailId();
        folderId = mail.getFolderId();
        flags = new MailFlagsCto(mail.getFlags());
        color = mail.getColor();
        receivedDate = mail.getReceivedDate();
        from = ListUtil.map(mail.getFrom(), contactsMapFunction);
        to = ListUtil.map(mail.getTo(), contactsMapFunction);
        cc = ListUtil.map(mail.getCc(), contactsMapFunction);
        bcc = ListUtil.map(mail.getBcc(), contactsMapFunction);
        subject = mail.getSubject();
        teaser = mail.getTeaser();
        messageId = mail.getMessageId();
        replyTo = ListUtil.map(mail.getReplyTo(), contactsMapFunction);
        sentDate = mail.getSentDate();
        contentType = mail.getContentType();
        size = mail.getSize();
        priority = getPriorityOrNull(mail.getPriority());
        List<AttachmentCto> attachments = ListUtil.map(mail.getParts(), partMapFunction);
        body = getBodyOrNull(attachments);
        bodyType = getBodyTypeOrNull(attachments);
        hasAttachment = mail.getHasAttachments() == Boolean.TRUE ? Boolean.TRUE : null;
        realAttachment = hasAttachment;
        this.attachments = getAttachmentListWithoutBodyOrNull(attachments);
    }

    private String getPriorityOrNull(Priority priority) {
        return priority != null ? priority.getStrValue() : null;
    }

    private String getBodyOrNull(List<AttachmentCto> attachments) {
        AttachmentCto part = findContentPart(attachments);
        return part != null ? part.getContent() : null;
    }

    private String getBodyTypeOrNull(List<AttachmentCto> attachments) {
        AttachmentCto part = findContentPart(attachments);
        return part != null ? Strings.toLowerCase(part.getContentType()) : null;
    }

    private AttachmentCto findContentPart(List<AttachmentCto> parts) {
        if (parts == null) {
            return null;
        }
        for (AttachmentCto part : parts) {
            String contentType = Strings.toLowerCase(part.getContentType());
            //if (!"attachment".equals(part.getDisposition())) {
            if (!("attachment".equals(part.getDisposition()) ||
                    ("inline".equals(part.getDisposition()) && !("text/plain".equals(contentType) ||"text/html".equals(contentType))))) {
                return part;
            }
        }
        return null;
    }

    private List<AttachmentCto> getAttachmentListWithoutBodyOrNull(List<AttachmentCto> attachments) {
        if (ListUtil.isEmpty(attachments)) {
            return null;
        }
        for (Iterator<AttachmentCto> it = attachments.iterator(); it.hasNext(); ) {
            AttachmentCto attachment = it.next();
            String contentType = Strings.toLowerCase(attachment.getContentType());
            //if (!"attachment".equals(attachment.getDisposition())) {
            if (!("attachment".equals(attachment.getDisposition()) ||
                    ("inline".equals(attachment.getDisposition()) && !("text/plain".equals(contentType) ||"text/html".equals(contentType))))) {
                it.remove();
            }
        }
        return attachments;
    }

}
