/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.configuration.clientspecific;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.utils.UserAgentInfo;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;

@Data
@EqualsAndHashCode(callSuper = true)
@Slf4j
public class MatchingHostConfiguration extends HostConfiguration {

    private List<MatchCriteria> criteria = new LinkedList<>();

    public MatchingHostConfiguration(HostConfiguration defaultConfiguration) {
        super(defaultConfiguration);
    }

    @Override
    public void loadProperties(Map<String, Object> properties) {
        super.loadProperties(properties);
        Object rawMatches = properties.get(MATCHES);
        if (rawMatches instanceof Map) {
            @SuppressWarnings("unchecked")
            Map<String, Object> matches = (Map<String, Object>) rawMatches;
            Iterator<Map.Entry<String, Object>> it = matches.entrySet().iterator();
            while (it.hasNext()) {
                Entry<String, Object> entry = it.next();
                String matcherName = entry.getKey();
                Object value = entry.getValue();
                switch (matcherName) {
                case "brand":
                    criteria.add(new BrandMatchCriteria((String) value));
                    continue;

                case "device":
                    criteria.add(new DeviceMatchCriteria((String) value));
                    continue;

                case "platform":
                    criteria.add(new PlatformMatchCriteria((String) value));
                    continue;

                case "version": {
                        VersionRange range = new VersionRange((String) value);
                        criteria.add(new AppVersionMatchCriteria(range));
                    }
                    continue;

                case "osVersion": {
                        VersionRange range = new VersionRange((String) value);
                        criteria.add(new OsVersionMatchCriteria(range));
                    }
                    continue;

                default:
                    log.warn("Unrecognized matcher name '" + matcherName + "'");
                    continue;
                }
            }
        }
    }

    public boolean matches(UserAgentInfo userAgentInfo) {
        Iterator<MatchCriteria> it = this.criteria.iterator();
        while (it.hasNext()) {
            MatchCriteria criteria = it.next();
            if (!criteria.matches(userAgentInfo)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public String toString() {
        return "matcher criteria: " + criteria;
    }

}
