/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import java.util.regex.Pattern;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.AuthenticationService;
import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.configuration.ConfigurationHelper;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.endpoints.requests.AuthenticationRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.LogoutRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.PushRequestBody;
import com.openexchange.mobile.api.facade.endpoints.responses.AuthenticationResponseBody;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.osgi.MultipleAuthenticationServicesHandler;
import com.openexchange.mobile.api.facade.services.ConfigService;
import com.openexchange.mobile.api.facade.utils.CookieUtil;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Path("/api-facade/v1/auth")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class AuthenticationEndpoint {

    private static final String FIELD_TOKEN = "token";

    private static final String FIELD_TRANSPORT = "transport";

    private static final Pattern PATTERN_CLIENT_ID = Pattern.compile("^\\p{Alnum}*$");

    private final MultipleAuthenticationServicesHandler authenticationHandler;

    private final ConfigService configService;

    @POST
    @Path("/login")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response authenticate(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Authenticating client");
        AuthenticationRequestBody authenticationRequestBody = JacksonUtil.parse(rawRequestBody, AuthenticationRequestBody.class);
        validate(authenticationRequestBody);
        String hostname = RequestUtil.getHostHeader(request);
        String userAgent = RequestUtil.getUserAgent(request);
        HostConfiguration configuration = ConfigurationHelper.getConfiguration(hostname);
        RequestUtil.checkUserAgent(userAgent, configuration);
        LoginCredentials credentials = getCredentials(request, authenticationRequestBody)
                .withHost(hostname)
                .withUserAgent(userAgent);
        AuthenticationService authenticationService = authenticationHandler.getAuthenticationService(configuration);
        SessionData sessionData = authenticationService.authenticate(credentials);
        sessionData = RequestUtil.enhanceSessionData(request, configuration, hostname, userAgent, sessionData);
        sessionData = configService.addConfiguration(sessionData);
        return buildResponse(sessionData);
    }

    void validate(AuthenticationRequestBody authenticationRequestBody) {
        if (authenticationRequestBody == null) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.INTERNAL_SERVER_ERROR, "Request body missing");
        }
        String clientId = authenticationRequestBody.getClientId();
        if (clientId != null && !PATTERN_CLIENT_ID.matcher(clientId).matches()) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.INTERNAL_SERVER_ERROR, "Invalid client ID");
        }
    }

    private LoginCredentials getCredentials(HttpServletRequest request, AuthenticationRequestBody requestBody) {
        String username = requestBody.getUsername();
        String password = requestBody.getPassword();
        String clientId = requestBody.getClientId();
        if (Strings.isEmpty(username)) {
            throw ApiFacadeException.missingParameter("username");
        }
        if (Strings.isEmpty(password)) {
            throw ApiFacadeException.missingParameter("password");
        }
        if ("".equals(clientId)) {
            clientId = null;
        }
        return new LoginCredentials(username, password, clientId)
                .withForwardedFor(RequestUtil.getForwardedForWithRemoteIp(request))
                .withRemoteIp(RequestUtil.getRemoteIp(request));
    }

    @POST
    @Path("/logout")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response logout(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Logging out client");
        SessionData sessionData = RequestUtil.getSessionData(request);
        HostConfiguration configuration = sessionData.getConfiguration();
        AuthenticationService authenticationService = authenticationHandler.getAuthenticationService(configuration);
        LogoutRequestBody requestBody = Strings.isEmpty(rawRequestBody)
                ? new LogoutRequestBody(null)
                : JacksonUtil.parse(rawRequestBody, LogoutRequestBody.class);
        validateLogoutRequestBody(requestBody);
        authenticationService.logout(sessionData, requestBody);
        return RequestUtil.emptyJsonResponse();
    }

    private void validateLogoutRequestBody(LogoutRequestBody requestBody) {
        PushRequestBody pushInfo = requestBody.getPushInfo();
        if (pushInfo == null) {
            return;
        }
        RequestUtil.checkParam(FIELD_TOKEN, pushInfo.getToken());
        RequestUtil.checkParam(FIELD_TRANSPORT, pushInfo.getTransport());
    }

    private Response buildResponse(SessionData sessionData) {
        AuthenticationResponseBody responseBody = new AuthenticationResponseBody(sessionData);
        String rawResponseBody = JacksonUtil.toString(responseBody);
        Response.ResponseBuilder builder = Response.ok(rawResponseBody);
        CookieUtil.addToResponse(builder, sessionData.getCookies());
        return builder.build();
    }

}
