/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.services;

import java.util.List;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.DeleteMailsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.EnvelopesConnector;
import com.openexchange.mobile.api.facade.connectors.impl.GetMailConnector;
import com.openexchange.mobile.api.facade.connectors.impl.MoveMailsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.SetAndClearFlagsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.SetColorsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.StubsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ThreadReferencesConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ThreadedAllStubsConnector;
import com.openexchange.mobile.api.facade.connectors.responses.ArrayDataResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.DeleteMailsResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.MailIdResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.MailResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ThreadReferencesResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ThreadedAllResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.ThreadMto;
import com.openexchange.mobile.api.facade.models.DeletedMails;
import com.openexchange.mobile.api.facade.models.Mail;
import com.openexchange.mobile.api.facade.models.MailFlags;
import com.openexchange.mobile.api.facade.models.MailIds;
import com.openexchange.mobile.api.facade.models.MailThread;
import com.openexchange.mobile.api.facade.utils.AttachmentUtil;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MailTransformations;
import com.openexchange.mobile.api.facade.utils.MailUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class MailsService {

    private final ConnectorFactoryFactory connectorFactoryFactory;

    private final MapFunction<ThreadMto, MailThread> threadMapFunction = new MapFunction<ThreadMto, MailThread>() {
        @Override
        public MailThread map(ThreadMto threadMto) {
            return new MailThread(threadMto);
        }
    };

    public List<MailThread> getStubs(SessionData sessionData, String folderId, boolean threaded) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        List<MailThread> mailThreads;
        ThreadingType threadingType = threaded ? sessionData.getConfiguration().getThreadingType() : ThreadingType.DISABLED;
        switch (threadingType) {
        case THREADREFERENCES:
            mailThreads = getStubsWithThreadReferences(connectorFactory, sessionData, folderId);
            break;

        case THREADEDALL:
            mailThreads = getStubsWithThreadedAll(connectorFactory, sessionData, folderId);
            break;

        case ALL:
            mailThreads = getStubsWithAll(connectorFactory, sessionData, folderId);
            break;

        case DISABLED:
        default:
            mailThreads = getStubsWithoutThreading(connectorFactory, sessionData, folderId);
            break;
        }
        return MailUtil.removeDeletedMails(mailThreads);
    }

    private List<MailThread> getStubsWithThreadReferences(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        if (folderId.startsWith("default0")) {
            ThreadReferencesConnector connector = connectorFactory.getThreadReferencesConnector(sessionData, folderId);
            ThreadReferencesResponseMto responseBody = connector.execute();
            return MailUtil.convert(responseBody.getData());
        } else {
            return getStubsWithoutThreading(connectorFactory, sessionData, folderId);
        }
    }

    private List<MailThread> getStubsWithAll(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        StubsConnector connector = connectorFactory.getStubsConnector(sessionData, folderId, true);
        ArrayDataResponseMto responseBody = connector.execute();
        List<Mail> mails = StubsConnector.map(responseBody.getData());
        return MailUtil.calculateThreads(mails);
    }

    private List<MailThread> getStubsWithThreadedAll(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        ThreadedAllStubsConnector connector = connectorFactory.getThreadedAllStubsConnector(sessionData, folderId);
        ThreadedAllResponseMto responseBody = connector.execute();
        return ListUtil.map(responseBody.getData(), threadMapFunction);
    }

    private List<MailThread> getStubsWithoutThreading(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        StubsConnector connector = connectorFactory.getStubsConnector(sessionData, folderId, false);
        ArrayDataResponseMto responseBody = connector.execute();
        List<Mail> mails = StubsConnector.map(responseBody.getData());
        return MailUtil.wrapAsThreads(mails);
    }

    public List<Mail> getEnvelopes(SessionData sessionData, String folderId, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        EnvelopesConnector connector = connectorFactory.getEnvelopesConnector(sessionData, folderId, mailIds);
        ArrayDataResponseMto responseBody = connector.execute();
        return EnvelopesConnector.map(responseBody.getData());
    }

    public Mail getMail(SessionData sessionData, String folderId, String mailId, int maximumTeaserLength) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        GetMailConnector connector = connectorFactory.getGetMailConnector(sessionData, folderId, mailId);
        MailResponseMto responseBody = connector.execute();
        Mail mail = GetMailConnector.map(responseBody.getData());
        boolean mailTeasersEnabled = sessionData.getConfiguration().isMailTeasersEnabled();
        mail = MailTransformations.apply(mail, mailTeasersEnabled, maximumTeaserLength);
        AttachmentUtil.initializeDownloadUrls(folderId, mail);
        return mail;
    }

    public DeletedMails delete(SessionData sessionData, String folderId, List<String> mailIds, boolean purge) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        DeleteMailsConnector connector = connectorFactory.getDeleteMailsConnector(sessionData, folderId, mailIds, purge);
        DeleteMailsResponseMto response = connector.execute();
        return DeleteMailsConnector.map(response);
    }

    public MailIds moveMails(SessionData sessionData, String folderId, List<String> mailIds, String targetFolderId) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        MoveMailsConnector connector = connectorFactory.getMoveMailsConnector(sessionData, folderId, mailIds, targetFolderId);
        List<MailIdResponseMto> response = connector.execute();
        List<String> newMailIds = MoveMailsConnector.map(response);
        return new MailIds(newMailIds);
    }

    public void setFlags(SessionData sessionData, String folderId, MailFlags flags, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        SetAndClearFlagsConnector connector = connectorFactory.getSetAndClearFlagsConnector(sessionData, folderId, mailIds, flags, true);
        connector.execute();
    }

    public void clearFlags(SessionData sessionData, String folderId, MailFlags flags, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        SetAndClearFlagsConnector connector = connectorFactory.getSetAndClearFlagsConnector(sessionData, folderId, mailIds, flags, false);
        connector.execute();
    }

    public void setColors(SessionData sessionData, String folderId, Integer color, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        SetColorsConnector connector = connectorFactory.getSetColorsConnector(sessionData, folderId, mailIds, color);
        connector.execute();
    }

}
