/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.openexchange.mobile.api.facade.models.MailContact;

import lombok.experimental.UtilityClass;

@UtilityClass
public class MailContactUtil {

    private static final Pattern PATTERN_BEGINS_WITH_QUOTE = Pattern.compile("^[\\\"'\\\\\\s]");

    private static final Pattern PATTERN_REORDER_DISPLAY_NAME = Pattern.compile("^([^,.\\(\\)]+),\\s([^,.\\(\\)]+)$");

    public List<MailContact> convert(List<List<String>> rawList) {
        if (ListUtil.isEmpty(rawList)) {
            return null;
        }

        List<MailContact> mailContacts = new ArrayList<>(rawList.size() / 2);

        for (List<String> entryList : rawList) {
            if (entryList.size() != 2) {
                throw new IllegalArgumentException("Entry list has not 2 elements");
            }

            String name = sanitizeName(entryList.get(0));
            String address = entryList.get(1);
            if (name != null) {
                name = reorderDisplayName(name);
            }
            if (address != null && address.contains("@")) {
                MailContact mailContact = new MailContact(name, address);
                mailContacts.add(mailContact);
            }
        }

        return mailContacts;
    }

    public String sanitizeName(String name) {
        if (name != null) {
            // Remove outer quotes.
            while (name.length() > 1
                    && PATTERN_BEGINS_WITH_QUOTE.matcher(name).lookingAt()
                    && name.charAt(0) == name.charAt(name.length() - 1)) {
                name = name.substring(1, name.length() - 1).trim();
            }

            // Unescape inner quotes.
            name = name.replaceAll("\\\"", "\"");

            // Unescape escaped backslashes.
            name = name.replaceAll("\\{2}", "\\");

            if (name.length() == 0) {
                name = null;
            }
        }
        return name;
    }

    public String reorderDisplayName(String name) {
        Matcher matcher = PATTERN_REORDER_DISPLAY_NAME.matcher(name);
        if (matcher.matches()) {
            name = matcher.group(2) + " " + matcher.group(1);
        }
        return name;
    }

}
