/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.endpoints.responses.ctos;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.openexchange.mobile.api.facade.models.Folder;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.NonNull;
import lombok.Value;

@Value
@JsonInclude(JsonInclude.Include.NON_NULL)
public class MailAccountCto {

    private static final MapFunction<String, AliasCto> aliasesMapFunction = new MapFunction<String, AliasCto>() {
        @Override
        public AliasCto map(String address) {
            return new AliasCto(address);
        }
    };

    // General properties

    String id;

    String accountName;

    String personalName;

    String login;

    String password;

    String userName;

    String firstName;

    String lastName;

    String primaryAddress;

    List<AliasCto> aliases;

    // IMAP/POP3 related properties

    String mailProtocol;

    String mailServer;

    Integer mailPort;

    Boolean mailSecure;

    Boolean mailStartTLS;

    Integer mailOAuth;

    Boolean mailDisabled;

    Integer pop3RefreshRate;

    Boolean pop3ExpungeOnQuit;

    Boolean pop3DeleteWriteThrough;

    // Transport related properties

    String transportAuth;

    String transportProtocol;

    String transportServer;

    Integer transportPort;

    Boolean transportSecure;

    Boolean transportStartTLS;

    String transportLogin;

    Integer transportOAuth;

    Boolean transportDisabled;

    // Folder related properties

    String spamHandler;

    String rootFolder;

    String trashFolder;

    String sentFolder;

    String draftsFolder;

    String spamFolder;

    String confirmedSpamFolder;

    String confirmedHamFolder;

    String inboxFolderId;

    String archiveFolderId;

    Boolean unifiedInboxEnabled;

    @NonNull
    private final List<FolderCto> folders;

    @NonNull
    private final StandardFoldersCto standardFolders;

    @JsonInclude(Include.NON_DEFAULT)
    Boolean canSend;

    public MailAccountCto(MailAccountData account) {
        id = account.getId();
        accountName = account.getAccountName();
        personalName = account.getPersonalName();
        login = account.getLogin();
        password = account.getPassword();
        userName = account.getUserName();
        firstName = account.getFirstName();
        lastName = account.getLastName();
        primaryAddress = account.getPrimaryAddress();
        aliases = ListUtil.map(account.getAliases(), aliasesMapFunction);

        mailProtocol = account.getMailProtocol();
        mailServer = account.getMailServer();
        mailPort = account.getMailPort();
        mailSecure = account.getMailSecure();
        mailStartTLS = account.getMailStartTLS();
        mailOAuth = account.getMailOAuth();
        mailDisabled = account.getMailDisabled();
        pop3RefreshRate = account.getPop3RefreshRate();
        pop3ExpungeOnQuit = account.getPop3ExpungeOnQuit();
        pop3DeleteWriteThrough = account.getPop3DeleteWriteThrough();

        transportAuth = account.getTransportAuth();
        transportProtocol = account.getTransportProtocol();
        transportServer = account.getTransportServer();
        transportPort = account.getTransportPort();
        transportSecure = account.getTransportSecure();
        transportStartTLS = account.getTransportStartTLS();
        transportLogin = account.getTransportLogin();
        transportOAuth = account.getTransportOAuth();
        transportDisabled = account.getTransportDisabled();

        spamHandler = account.getSpamHandler();
        rootFolder = account.getRootFolderId();
        trashFolder = account.getTrashFolder();
        sentFolder = account.getSentFolder();
        draftsFolder = account.getDraftsFolder();
        spamFolder = account.getSpamFolder();
        confirmedSpamFolder = account.getConfirmedSpamFolder();
        confirmedHamFolder = account.getConfirmedHamFolder();
        inboxFolderId = account.getInboxFolderId();
        archiveFolderId = account.getArchiveFolderId();
        unifiedInboxEnabled = account.getUnifiedInboxEnabled();

        canSend = account.getCanSend();

        if (account.getFolders() != null) {
            /* Java8: Use with Java 8.
            folders = account.getFolders().stream()
                    .map(folder -> new FolderCto(folder))
                    .collect(Collectors.toList());
            */
            folders = new ArrayList<>();
            for (Folder folder : account.getFolders()) {
                folders.add(new FolderCto(folder));
            }
            standardFolders = new StandardFoldersCto(account.getStandardFolders());
        } else {
            folders = null;
            standardFolders = null;
        }
    }

}
