/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.configuration;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import com.openexchange.config.ConfigurationService;
import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.impl.DefaultAuthenticationService;
import com.openexchange.mobile.api.facade.services.ThreadingType;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import org.apache.http.util.TextUtils;

@Data
@Slf4j
public class HostConfiguration {

    /*private*/ static final String AUTHENTICATION_CLASS_NAME = "com.openexchange.mobile.api.facade.authenticationClass";

    /*private*/ static final String MINIMUM_CLIENT_VERSION_ANDROID = "com.openexchange.mobile.api.facade.minimumClientVersion.android";

    /*private*/ static final String MINIMUM_CLIENT_VERSION_IOS = "com.openexchange.mobile.api.facade.minimumClientVersion.ios";

    /*private*/ static final String MINIMUM_CLIENT_VERSION_WINDOWS = "com.openexchange.mobile.api.facade.minimumClientVersion.windows";

    /*private*/ static final String THREADING_TYPE = "com.openexchange.mobile.api.facade.threading";

    /*private*/ static final String MAILS_PER_FOLDER = "com.openexchange.mobile.api.facade.numMailsPerFolder";

    /*private*/ public static final String RETURN_NON_PRIMARY_ACCOUNTS = "com.openexchange.mobile.api.facade.returnNonPrimaryAccounts";

    /*private*/ static final String INCLUDE_NON_PRIVATE_FOLDERS = "com.openexchange.mobile.api.facade.includeNonPrivateFolders";

    /*private*/ static final String MAIL_TEASERS_ENABLED = "com.openexchange.mobile.api.facade.mailTeasers";

    /*private*/ static final String MAIL_TEASERS_DEFAULT_LENGTH = "com.openexchange.mobile.api.facade.mailTeasers.defaultLength";

    /*private*/ public static final String PLATFORM_ANDROID = "Android";

    /*private*/ public static final String PLATFORM_IOS = "iOS";

    /*private*/ public static final String PLATFORM_WINDOWS = "Windows";

    /*private*/ public static final String WEAKFORCE_ENABLED = "com.openexchange.mobile.api.facade.weakforce.enabled";

    /*private*/ public static final String WEAKFORCE_BASE_URL = "com.openexchange.mobile.api.facade.weakforce.baseUrl";

    /*private*/ public static final String WEAKFORCE_PASSWORD = "com.openexchange.mobile.api.facade.weakforce.password";

    /*private*/ public static final String WEAKFORCE_HASH_MECHANISM = "com.openexchange.mobile.api.facade.weakforce.hashMechanism";

    /*private*/ public static final String WEAKFORCE_HASH_NONCE = "com.openexchange.mobile.api.facade.weakforce.hashNonce";

    /*private*/ public static final String ALLOWED_BRANDS = "com.openexchange.mobile.api.facade.allowedBrands";

    /*private*/ public static final String FORWARD_HOST_HEADER = "com.openexchange.mobile.api.facade.forwardHostHeader";

    /*private*/ public static final String CUSTOM_PROPERTIES = "customProperties";

    String authenticationClassName;

    Map<String, String> minimumClientVersions = new Hashtable<>();

    Map<String, String> customProperties = new HashMap<>();

    ThreadingType threadingType;

    String numMailsPerFolder;

    boolean returnNonPrimaryAccounts;

    boolean includeNonPrivateFolders;

    boolean mailTeasersEnabled;

    int mailTeasersDefaultLength;

    boolean weakforceEnabled;

    String weakforceBaseUrl;

    String weakforcePassword;

    boolean weakforceUseAuthentication;

    String weakforceHashMechanism;

    String weakforceHashNonce;

    List<String> allowedBrands;

    boolean forwardHostHeader;

    public HostConfiguration(HostConfiguration otherConfiguration) {
        authenticationClassName = otherConfiguration.authenticationClassName;
        minimumClientVersions.putAll(otherConfiguration.minimumClientVersions);
        threadingType = otherConfiguration.threadingType;
        numMailsPerFolder = otherConfiguration.numMailsPerFolder;
        returnNonPrimaryAccounts = otherConfiguration.returnNonPrimaryAccounts;
        includeNonPrivateFolders = otherConfiguration.includeNonPrivateFolders;
        mailTeasersEnabled = otherConfiguration.mailTeasersEnabled;
        mailTeasersDefaultLength = otherConfiguration.mailTeasersDefaultLength;
        weakforceEnabled = otherConfiguration.weakforceEnabled;
        weakforceBaseUrl = otherConfiguration.weakforceBaseUrl;
        weakforcePassword = otherConfiguration.weakforcePassword;
        weakforceUseAuthentication = otherConfiguration.weakforceUseAuthentication;
        weakforceHashMechanism = otherConfiguration.weakforceHashMechanism;
        weakforceHashNonce = otherConfiguration.weakforceHashMechanism;
        allowedBrands = otherConfiguration.allowedBrands;
        forwardHostHeader = otherConfiguration.forwardHostHeader;
    }

    public HostConfiguration(ConfigurationService configurationService) {
        authenticationClassName = ConfigurationHelper.getString(configurationService, AUTHENTICATION_CLASS_NAME, DefaultAuthenticationService.class.getName());
        loadMinimumVersions(configurationService);
        threadingType = ThreadingType.from(ConfigurationHelper.getString(configurationService, THREADING_TYPE, ThreadingType.THREADREFERENCES.toString()));
        numMailsPerFolder = ConfigurationHelper.getNumberString(configurationService, MAILS_PER_FOLDER, "10000");
        returnNonPrimaryAccounts = ConfigurationHelper.getBoolean(configurationService, RETURN_NON_PRIMARY_ACCOUNTS, false);
        includeNonPrivateFolders = ConfigurationHelper.getBoolean(configurationService, INCLUDE_NON_PRIVATE_FOLDERS, false);
        mailTeasersEnabled = ConfigurationHelper.getBoolean(configurationService, MAIL_TEASERS_ENABLED, true);
        mailTeasersDefaultLength = ConfigurationHelper.getNumber(configurationService, MAIL_TEASERS_DEFAULT_LENGTH, 300);
        weakforceEnabled = ConfigurationHelper.getBoolean(configurationService, WEAKFORCE_ENABLED, false);
        weakforceBaseUrl = ConfigurationHelper.getString(configurationService, WEAKFORCE_BASE_URL, null);
        weakforcePassword = ConfigurationHelper.getString(configurationService, WEAKFORCE_PASSWORD, null);
        weakforceUseAuthentication = !Strings.isEmpty(weakforcePassword);
        weakforceHashMechanism = ConfigurationHelper.getString(configurationService, WEAKFORCE_HASH_MECHANISM, null);
        weakforceHashNonce = ConfigurationHelper.getString(configurationService, WEAKFORCE_HASH_NONCE, null);
        String allowedBrandsStr = ConfigurationHelper.getStringOrNull(configurationService, ALLOWED_BRANDS);
        allowedBrands = allowedBrands != null ? Strings.splitAndTrim(allowedBrandsStr, ",") : null;
        forwardHostHeader = ConfigurationHelper.getBoolean(configurationService, FORWARD_HOST_HEADER, true);
    }

    private void loadMinimumVersions(ConfigurationService configurationService) {
        minimumClientVersions.clear();
        addMinimumVersionFor(configurationService, PLATFORM_ANDROID, MINIMUM_CLIENT_VERSION_ANDROID);
        addMinimumVersionFor(configurationService, PLATFORM_IOS, MINIMUM_CLIENT_VERSION_IOS);
        addMinimumVersionFor(configurationService, PLATFORM_WINDOWS, MINIMUM_CLIENT_VERSION_WINDOWS);
    }

    private void addMinimumVersionFor(ConfigurationService configurationService, String platform, String key) {
        String minimumClientVersion = ConfigurationHelper.getString(configurationService, key, null);
        if (!TextUtils.isEmpty(minimumClientVersion)) {
            minimumClientVersions.put(platform, minimumClientVersion);
        }
    }

    public void loadProperties(Map<String, Object> properties) {
        for (Map.Entry<String, Object> entry : properties.entrySet()) {
            String propertyName = entry.getKey();
            Object propertyValue = entry.getValue();
            String minimalVersionValue;
            String platformName = getPlatformForBrandedMinimumClientVersionProperty(propertyName);
            if (platformName != null) {
                minimalVersionValue = getString(propertyValue, null);
                if (minimalVersionValue != null) {
                    minimumClientVersions.put(platformName, minimalVersionValue);
                }
            }
            switch (propertyName) {
            case THREADING_TYPE:
                threadingType = ThreadingType.from(getString(propertyValue, ThreadingType.THREADREFERENCES.toString()));
                continue;

            case MAILS_PER_FOLDER:
                numMailsPerFolder = getNumberString(propertyValue, "10000");
                continue;

            case RETURN_NON_PRIMARY_ACCOUNTS:
                returnNonPrimaryAccounts = getBoolean(propertyValue, false);
                continue;

            case INCLUDE_NON_PRIVATE_FOLDERS:
                includeNonPrivateFolders = getBoolean(propertyValue, false);
                continue;

            case MAIL_TEASERS_ENABLED:
                mailTeasersEnabled = getBoolean(propertyValue, true);
                continue;

            case MAIL_TEASERS_DEFAULT_LENGTH:
                mailTeasersDefaultLength = getNumber(propertyValue, 300);
                continue;

            case MINIMUM_CLIENT_VERSION_ANDROID:
                minimalVersionValue = getString(propertyValue, null);
                if (minimalVersionValue != null) {
                    minimumClientVersions.put(PLATFORM_ANDROID, minimalVersionValue);
                }
                continue;

            case MINIMUM_CLIENT_VERSION_IOS:
                minimalVersionValue = getString(propertyValue, null);
                if (minimalVersionValue != null) {
                    minimumClientVersions.put(PLATFORM_IOS, minimalVersionValue);
                }
                continue;

            case MINIMUM_CLIENT_VERSION_WINDOWS:
                minimalVersionValue = getString(propertyValue, null);
                if (minimalVersionValue != null) {
                    minimumClientVersions.put(PLATFORM_WINDOWS, minimalVersionValue);
                }
                continue;

            case ALLOWED_BRANDS:
                String allowedBrandsValue = getString(propertyValue, null);
                allowedBrands = Strings.splitAndTrim(allowedBrandsValue, ",");
                if (allowedBrands.size() == 0) {
                    allowedBrands = null;
                }
                continue;

            case CUSTOM_PROPERTIES:
                @SuppressWarnings("unchecked")
                Map<String, Object> map = (Map<String, Object>) propertyValue;
                for (Map.Entry<String, Object> customPropertyEntry : map.entrySet()) {
                    customProperties.put(customPropertyEntry.getKey(), customPropertyEntry.getValue().toString());
                }
                continue;

            default:
                log.warn("Unrecognized configuration property '" + propertyName + "'");
                continue;
            }
        }
    }

    private String getPlatformForBrandedMinimumClientVersionProperty(String propertyName) {
        String platformName = null;
        if (propertyName.startsWith(MINIMUM_CLIENT_VERSION_ANDROID)) {
            platformName = getPlatformForBrandedMinimumClientVersionProperty(propertyName, MINIMUM_CLIENT_VERSION_ANDROID, PLATFORM_ANDROID);
        } else if (propertyName.startsWith(MINIMUM_CLIENT_VERSION_IOS)) {
            platformName = getPlatformForBrandedMinimumClientVersionProperty(propertyName, MINIMUM_CLIENT_VERSION_IOS, PLATFORM_IOS);
        } else if (propertyName.startsWith(MINIMUM_CLIENT_VERSION_WINDOWS)) {
            platformName = getPlatformForBrandedMinimumClientVersionProperty(propertyName, MINIMUM_CLIENT_VERSION_WINDOWS, PLATFORM_WINDOWS);
        }
        return platformName;
    }

    private String getPlatformForBrandedMinimumClientVersionProperty(String propertyName, String platformPropertyName, String platformName) {
        if (propertyName.length() > platformPropertyName.length() + 1) {
            platformName += "." + propertyName.substring(platformPropertyName.length() + 1);
        }
        return platformName;
    }

    private String getString(Object value, String defaultValue) {
        if (value instanceof String) {
            return (String) value;
        } else {
            log.warn(String.format("Invalid string value '%s'. Using default value '%s'", value, defaultValue));
        }
        return defaultValue;
    }

    private String getNumberString(Object value, String defaultValue) {
        try {
            if (value instanceof String) {
                String strValue = (String) value;
                Integer.parseInt(strValue);
                return strValue;
            } else {
                Integer intValue = (Integer) value;
                return intValue.toString();
            }
        } catch (Exception e) {
            log.warn(String.format("Invalid string value '%s'. Using default value '%s'", value, defaultValue));
        }
        return defaultValue;
    }

    private int getNumber(Object value, int defaultValue) {
        if (value instanceof Integer) {
            return ((Integer) value).intValue();
        } else {
            log.warn(String.format("Invalid integer value '%s'. Using default value '%d'", value, defaultValue));
        }
        return defaultValue;
    }

    private boolean getBoolean(Object value, boolean defaultValue) {
        if (value instanceof Boolean) {
            return ((Boolean) value).booleanValue();
        } else {
            log.warn(String.format("Invalid boolean value '%s'. Using default value '%s'", value, defaultValue));
        }
        return defaultValue;
    }

    public boolean isAllowedBrand(String brandName) {
        return allowedBrands == null || allowedBrands.contains(brandName);
    }

}
