/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.services;

import java.util.Collections;
import java.util.List;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.JsLob;
import com.openexchange.mobile.api.facade.connectors.impl.AccountsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.AutoconfigureAccountConnector;
import com.openexchange.mobile.api.facade.connectors.impl.CreateAccountConnector;
import com.openexchange.mobile.api.facade.connectors.impl.DeleteAccountConnector;
import com.openexchange.mobile.api.facade.connectors.impl.GetAccountConnector;
import com.openexchange.mobile.api.facade.connectors.impl.GetJsLobConnector;
import com.openexchange.mobile.api.facade.connectors.impl.MailAccountBaseConnector;
import com.openexchange.mobile.api.facade.connectors.impl.UpdateAccountConnector;
import com.openexchange.mobile.api.facade.connectors.impl.UserConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ValidateAccountConnector;
import com.openexchange.mobile.api.facade.connectors.responses.ArrayDataResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.BooleanResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.IntegerListResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.MailAccountResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.RawResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.UserResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.AccountMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.UserMto;
import com.openexchange.mobile.api.facade.endpoints.requests.AutoconfigData;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.models.Account;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.utils.AccountUtil;
import com.openexchange.mobile.api.facade.utils.JsonUtil;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class AccountsService {

    private static final String PRIMARY_ACCOUNT = "0";

    private static final String ERROR_CODE_ACCOUNT_DOESNT_EXIST = "ACC-0002";

    private static final MapFunction<AccountMto, String> accountsToIdsMapFunction = new MapFunction<AccountMto, String>() {

        @Override
        public String map(AccountMto accountMto) {
            return accountMto.getId();
        }

    };

    private final ConnectorFactoryFactory connectorFactoryFactory;

    private final FoldersService foldersService;

    public List<String> getAccountList(SessionData sessionData) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        List<AccountMto> accountMtos = retrieveAccounts(connectorFactory, sessionData);
        return ListUtil.map(accountMtos, accountsToIdsMapFunction);
    }

    public List<Account> getAccounts(SessionData sessionData) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        return getAccounts(connectorFactory, sessionData);
    }

    public List<Account> getAccounts(final ConnectorFactory connectorFactory, final SessionData sessionData) {
        List<AccountMto> accountMtos = retrieveAccounts(connectorFactory, sessionData);
        final UserMto userMto = retrieveUserMto(connectorFactory, sessionData);
        return ListUtil.map(accountMtos, new MapFunction<AccountMto, Account>() {

            @Override
            public Account map(AccountMto accountMto) {
                return mapAccount(connectorFactory, sessionData, accountMto, userMto);
            }

        });
    }

    public List<MailAccountData> getAccountsV2(SessionData sessionData, String language) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        return retrieveAccountsV2(connectorFactory, sessionData);
    }

    public Account getAccount(SessionData sessionData, String accountId, String language) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        Account account = getAccount(connectorFactory, sessionData, accountId);
        foldersService.addFolders(connectorFactory, sessionData, account, language);
        return account;
    }

    public MailAccountData getAccountV2(SessionData sessionData, String accountId, String language) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        UserMto userMto = retrieveUserMto(connectorFactory, sessionData);
        GetAccountConnector connector = connectorFactory.getAccountConnector(sessionData, accountId);
        MailAccountResponseMto response = connector.execute();
        MailAccountData account = GetAccountConnector.map(response);
        account.setUserName(AccountUtil.getUserName(account.getPersonalName(), userMto));
        account.setFirstName(userMto.getFirstName());
        account.setLastName(userMto.getLastName());
        foldersService.addFolders(connectorFactory, sessionData, account, language);
        return account;
    }

    public Account getAccount(final ConnectorFactory connectorFactory, final SessionData sessionData, String accountId) {
        AccountMto accountMto = retrieveAccount(connectorFactory, sessionData, accountId);
        if (accountMto == null) {
            return null;
        }
        final UserMto userMto = retrieveUserMto(connectorFactory, sessionData);
        return mapAccount(connectorFactory, sessionData, accountMto, userMto);
    }

    private Account mapAccount(ConnectorFactory connectorFactory, SessionData sessionData, AccountMto accountMto, UserMto userMto) {
        String defaultSendAddress = accountMto.isPrimaryAccount() ? retrieveDefaultSendAddress(connectorFactory, sessionData) : null;
        return new Account(accountMto, userMto, defaultSendAddress);
    }

    private String retrieveDefaultSendAddress(ConnectorFactory connectorFactory, SessionData sessionData) {
        GetJsLobConnector jsLobConnector = connectorFactory.getGetJsLobConnector(sessionData, JsLob.MAIL);
        RawResponseMto response = jsLobConnector.execute();
        return JsonUtil.getString(GetJsLobConnector.map(response), "defaultSendAddress");
    }

    private AccountMto retrieveAccount(ConnectorFactory connectorFactory, SessionData sessionData, String accountId) {
        AccountsConnector accountsConnector = connectorFactory.getAccountsConnector(sessionData);
        ArrayDataResponseMto response = accountsConnector.execute();
        Object[][] data = response.getData();
        if (data == null || data.length == 0) {
            throw ApiFacadeException.internalServerError("No account data available");
        }
        return AccountsConnector.map(data, accountId);
    }

    private List<AccountMto> retrieveAccounts(ConnectorFactory connectorFactory, SessionData sessionData) {
        AccountsConnector accountsConnector = connectorFactory.getAccountsConnector(sessionData);
        ArrayDataResponseMto response = accountsConnector.execute();
        Object[][] data = response.getData();
        if (data == null || data.length == 0) {
            throw ApiFacadeException.internalServerError("No account data available");
        }
        List<AccountMto> accounts = AccountsConnector.map(data);
        if (sessionData.getConfiguration().isReturnNonPrimaryAccounts()) {
            return accounts;
        }
        return Collections.singletonList(accounts.get(0));
    }

    private List<MailAccountData> retrieveAccountsV2(ConnectorFactory connectorFactory, SessionData sessionData) {
        AccountsConnector accountsConnector = connectorFactory.getAccountsConnector(sessionData);
        ArrayDataResponseMto response = accountsConnector.execute();
        Object[][] data = response.getData();
        if (data == null || data.length == 0) {
            throw ApiFacadeException.internalServerError("No account data available");
        }
        List<MailAccountData> accounts = AccountsConnector.mapToMailAccountData(data);
        if (sessionData.getConfiguration().isReturnNonPrimaryAccounts()) {
            return accounts;
        }
        return Collections.singletonList(accounts.get(0));
    }

    private UserMto retrieveUserMto(ConnectorFactory connectorFactory, SessionData sessionData) {
        UserConnector connector = connectorFactory.getUserConnector(sessionData);
        UserResponseMto response = connector.execute();
        return response.getData();
    }

    public MailAccountData createAccount(SessionData sessionData, MailAccountData account, boolean forceInsecureConnection) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        // 1. Validate account data.
        // 2. Throw exception indicating invalid account data if not successful validated.
        validateAccountData(connectorFactory, sessionData, account, forceInsecureConnection);
        // 3. If successfully validated create new account
        CreateAccountConnector connector = connectorFactory.getCreateAccountConnector(sessionData, account);
        MailAccountResponseMto response = connector.execute();
        // 4. When account is updated return new account data.
        return MailAccountBaseConnector.map(response);
    }

    public MailAccountData updateAccount(SessionData sessionData, MailAccountData account, boolean forceInsecureConnection) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        // 1. Validate account data.
        // 2. Throw exception indicating invalid account data if not successful validated.
        validateAccountData(connectorFactory, sessionData, account, forceInsecureConnection);
        // 3. If successfully validated update account.
        UpdateAccountConnector connector = connectorFactory.getUpdateAccountConnector(sessionData, account);
        MailAccountResponseMto response = connector.execute();
        // 4. When account is updated return new account data.
        return MailAccountBaseConnector.map(response);
    }

    private void validateAccountData(ConnectorFactory connectorFactory, SessionData sessionData, MailAccountData account, boolean forceInsecureConnection) {
        ValidateAccountConnector validateConnector = connectorFactory.getValidateAccountConnector(sessionData, account, forceInsecureConnection);
        BooleanResponseMto response = validateConnector.execute();
        if (ValidateAccountConnector.map(response)) {
            return;
        }
        throw ApiFacadeException.invalidAccountData();
    }

    public boolean deleteAccount(SessionData sessionData, String accountId) {
        if (PRIMARY_ACCOUNT.equals(accountId)) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.INVALID_ACCOUNT_DATA, "It's not allowed to delete the primary account");
        }
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        DeleteAccountConnector connector = connectorFactory.getDeleteAccountConnector(sessionData, accountId);
        try {
            IntegerListResponseMto response = connector.execute();
            return DeleteAccountConnector.map(response);
        } catch (ApiFacadeException e) {
            if (ERROR_CODE_ACCOUNT_DOESNT_EXIST.equals(e.getErrorCode())) {
                return false;
            }
            throw e;
        }
    }

    public MailAccountData autoconfigAccount(SessionData sessionData, AutoconfigData autoconfigData) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        AutoconfigureAccountConnector connector = connectorFactory.getAutoconfigureAccountConnector(sessionData, autoconfigData);
        MailAccountResponseMto response = connector.execute();
        return new MailAccountData(response.getData());
    }

}
