/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.services;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.AccountsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.AllFoldersConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ClearFolderConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ConfigConnector;
import com.openexchange.mobile.api.facade.connectors.impl.CreateFolderConnector;
import com.openexchange.mobile.api.facade.connectors.impl.DeleteFolderConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ExamineMultipleFoldersConnector;
import com.openexchange.mobile.api.facade.connectors.impl.UserConnector;
import com.openexchange.mobile.api.facade.connectors.responses.ArrayDataResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ConfigResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ExamineResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.FoldersResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.StringResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.UserResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.AccountMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.ExamineResultMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.UserMto;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.models.Account;
import com.openexchange.mobile.api.facade.models.Config;
import com.openexchange.mobile.api.facade.models.CreatedFolder;
import com.openexchange.mobile.api.facade.models.ExamineResult;
import com.openexchange.mobile.api.facade.models.Folder;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class FoldersService {

    private final ConnectorFactoryFactory connectorFactoryFactory;

    public CreatedFolder createFolder(SessionData sessionData, String parentFolderId, String name) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        CreateFolderConnector connector = connectorFactory.getCreateFolderConnector(sessionData, parentFolderId, name);
        StringResponseMto response = connector.execute();
        String folderId = CreateFolderConnector.map(response);
        return new CreatedFolder(folderId, name);
    }

    public void deleteFolder(SessionData sessionData, String folderId) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        DeleteFolderConnector connector = connectorFactory.getDeleteFolderConnector(sessionData, folderId);
        connector.execute();
    }

    public void clearFolder(SessionData sessionData, String folderId) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        ClearFolderConnector connector = connectorFactory.getClearFolderConnector(sessionData, folderId);
        connector.execute();
    }

    public List<Account> getAllMailFolders(final SessionData sessionData, final String language) {
        final ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        Config config = retrieveConfiguration(connectorFactory, sessionData);
        final boolean isNamespaceSet = "INBOX/".equals(config.getNamespace());
        List<AccountMto> accountMtos = retrieveAccounts(connectorFactory, sessionData);
        final UserMto userMto = retrieveUserMto(connectorFactory, sessionData);
        return ListUtil.map(accountMtos, new MapFunction<AccountMto, Account>() {

            @Override
            public Account map(AccountMto accountMto) {
                List<Folder> folders = retrieveFolders(connectorFactory, sessionData, accountMto, language);
                examineFolders(connectorFactory, sessionData, folders);
                Account account = new Account(accountMto, userMto, folders);
                adjustFolderParents(account, isNamespaceSet);
                return account;
            }

        });
    }

    private Config retrieveConfiguration(ConnectorFactory connectorFactory, SessionData sessionData) {
        ConfigConnector configConnector = connectorFactory.getConfigConnector(sessionData);
        ConfigResponseMto response = configConnector.execute();
        return ConfigConnector.map(response);
    }

    private List<AccountMto> retrieveAccounts(ConnectorFactory connectorFactory, SessionData sessionData) {
        AccountsConnector accountsConnector = connectorFactory.getAccountsConnector(sessionData);
        ArrayDataResponseMto response = accountsConnector.execute();
        Object[][] data = response.getData();
        if (data == null || data.length == 0) {
            throw ApiFacadeException.internalServerError("No account data available");
        }
        List<AccountMto> accounts = AccountsConnector.map(data);
        if (sessionData.getConfiguration().isReturnNonPrimaryAccounts()) {
            return accounts;
        }
        return Collections.singletonList(accounts.get(0));
    }

    private UserMto retrieveUserMto(ConnectorFactory connectorFactory, SessionData sessionData) {
        UserConnector connector = connectorFactory.getUserConnector(sessionData);
        UserResponseMto response = connector.execute();
        return response.getData();
    }

    private List<Folder> retrieveFolders(ConnectorFactory connectorFactory, SessionData sessionData, AccountMto account, String language) {
        boolean includeNonPrivateFolders = sessionData.getConfiguration().isIncludeNonPrivateFolders();
        AllFoldersConnector connector = connectorFactory.getAllFoldersConnector(sessionData, account.getRootFolderId(), language);
        FoldersResponseMto response = connector.execute();
        List<Folder> folders = new ArrayList<>();
        if (response.getData() != null) {
            for (Object[] data : response.getData().getPrivateFolders()) {
                Folder folder = AllFoldersConnector.map(data);
                if (isMailFolder(folder, includeNonPrivateFolders)) {
                    if (folder.getId().equals(account.getArchiveFolderId())) {
                        folder = folder.withType(Folder.Type.ARCHIVE);
                    }
                    folders.add(folder);
                }
            }
        }
        return folders;
    }

    private boolean isMailFolder(Folder folder, boolean includeNonPrivateFolders) {
        boolean isSubFolder = folder.getParentId() != null && folder.getId().startsWith(folder.getParentId());
        return isSubFolder && (includeNonPrivateFolders || (!Folder.Type.PUBLIC.equals(folder.getType()) && !Folder.Type.SHARED.equals(folder.getType())));
    }

    private void examineFolders(ConnectorFactory connectorFactory, SessionData sessionData, List<Folder> folders) {
        ExamineMultipleFoldersConnector connector = connectorFactory.getExamineMultipleFoldersConnector(sessionData, folders);
        List<ExamineResponseMto> responses = connector.execute();

        for (int index = 0; index < folders.size(); index++) {
            Folder folder = folders.get(index);
            ExamineResponseMto response = responses.get(index);
            ExamineResultMto result = response.getData();
            if (!response.isError() && result != null) {
                folder.setValidity(result.getValidity());
                folder.setNext(result.getNext());
                folder.setTotal(result.getTotal());
                folder.setUnread(result.getUnread());
            } else {
                folders.remove(index);
                index--;
            }
        }
    }

    public ExamineResult examineFolder(SessionData sessionData, String folderId, String validity, String next) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        ExamineMultipleFoldersConnector connector = connectorFactory.getExamineFolderConnector(sessionData, folderId);
        List<ExamineResponseMto> responses = connector.execute();
        ExamineResultMto response = responses.get(0).getData();
        String newValidity = response.getValidity();
        String newNext = response.getNext();
        boolean changed = !validity.equals(newValidity) || !next.equals(newNext);
        return new ExamineResult(folderId, changed, newValidity, newNext);
    }

    private void adjustFolderParents(Account account, boolean isNamespaceSet) {
        Folder inboxFolder = account.getStandardFolders().getInboxFolder();
        if (inboxFolder != null && isNamespaceSet) {
            String inboxFolderId = inboxFolder.getId();
            String parentFolderId = inboxFolder.getParentId();
            for (Folder folder : account.getFolders()) {
                if (inboxFolderId.equals(folder.getParentId())) {
                    folder.setParentId(parentFolderId);
                }
            }
        }
    }

}
