/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.auth.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.Connector;
import com.openexchange.mobile.api.facade.connectors.RequestConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;
import com.openexchange.mobile.api.facade.utils.StringUtils;

import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.CookieStore;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;

@Slf4j
public class LoginConnector implements Connector<SessionData, LoginResponseMto> {

    private final RequestConfiguration configuration;

    private final LoginCredentials credentials;

    public LoginConnector(RequestConfiguration configuration, LoginCredentials credentials) {
        this.configuration = configuration;
        this.credentials = credentials;
    }

    @Override
    public String createUrl() {
        return configuration.getBaseUrl() + "login";
    }

    @Override
    public SessionData execute() throws ApiFacadeException {
        try {
            CloseableHttpClient httpClient = configuration.getHttpClient();
            CookieStore cookieStore = configuration.getCookieStore();
            HttpUriRequest request = createRequest();
            CloseableHttpResponse response = httpClient.execute(request);
            LoginResponseMto responseBody = handleResponse(response);
            response.close();
            if (responseBody.isError()) {
                throw ApiFacadeException.fromResponseBody(responseBody);
            }
            if (StringUtils.isEmpty(responseBody.getSession())) {
                throw ApiFacadeException.internalServerError("No session received.");
            }
            if (cookieStore.getCookies().size() == 0) {
                throw ApiFacadeException.internalServerError("No cookies received.");
            }
            return new SessionData(responseBody.getSession(), cookieStore.getCookies());
        } catch (Exception e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

    private HttpUriRequest createRequest() {
        String clientId = "open-xchange-mobile-api-facade";
        if (credentials.getClientId() != null) {
            clientId += "-" + credentials.getClientId();
        }
        RequestBuilder builder = RequestBuilder.post(createUrl());
        builder.setCharset(Charset.forName("UTF-8"));
        String userAgent = credentials.getUserAgent();
        if (userAgent != null) {
            builder.addHeader(RequestUtil.HEADER_USER_AGENT, userAgent);
        }
        return builder
                .addParameter("action", "login")
                .addParameter("name", credentials.getUsername())
                .addParameter("password", credentials.getPassword())
                .addParameter("client", clientId)
                .build();
    }

    @Override
    public LoginResponseMto handleResponse(HttpResponse response) {
        try {
            StatusLine statusLine = response.getStatusLine();
            int statusCode = statusLine.getStatusCode();
            if (statusCode == 200) {
                HttpEntity entity = response.getEntity();
                if (log.isDebugEnabled()) {
                    entity = new BufferedHttpEntity(response.getEntity());
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    entity.writeTo(baos);
                    log.debug(new String(baos.toByteArray(), "UTF-8"));
                }
                ObjectMapper mapper = JacksonUtil.getMapper();
                LoginResponseMto responseBody = mapper.readValue(entity.getContent(), LoginResponseMto.class);
                EntityUtils.consume(entity);
                return responseBody;
            }
            throw ApiFacadeException.internalServerError("Unable to log in");
        } catch (IOException e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

}
