/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ws.rs.core.NewCookie;
import javax.ws.rs.core.Response;

import lombok.experimental.UtilityClass;

import org.apache.http.cookie.Cookie;
import org.apache.http.impl.cookie.BasicClientCookie;

@UtilityClass
public class CookieUtil {

    private static final int SECONDS_TWO_WEEKS = 1209600;

    private static final int MILLIS_ONE_WEEK = 604800000;

    public void addToResponse(Response.ResponseBuilder builder, List<Cookie> cookies) {
        /* Java8: Use this with Java 8.
        convertForEndpoint(cookies)
                .forEach(cookie -> builder.cookie(cookie));
        */
        for (NewCookie cookie : convertForEndpoint(cookies)) {
            builder.cookie(cookie);
        }
    }

    public List<NewCookie> convertForEndpoint(List<Cookie> cookies) {
        /* Java8: Use this with Java 8.
        return cookies.stream()
                .map(cookie -> convertForEndpoint(cookie))
                .collect(toList());
        */
        List<NewCookie> result = new ArrayList<>();
        for (Cookie cookie : cookies) {
            NewCookie newCookie = convertForEndpoint(cookie);
            result.add(newCookie);
        }
        return result;
    }

    public NewCookie convertForEndpoint(Cookie cookie) {
        return new NewCookie(cookie.getName(), cookie.getValue(), cookie.getPath(), null, null, calculateMaxAge(cookie), cookie.isSecure());
    }

    private int calculateMaxAge(Cookie cookie) {
        int maxAge;
        Date expiresDate = cookie.getExpiryDate();
        if (expiresDate != null) {
            maxAge = (int) ((expiresDate.getTime() - System.currentTimeMillis()) / 1000);
        } else {
            maxAge = SECONDS_TWO_WEEKS;
        }
        return maxAge;
    }

    public List<Cookie> convertForConnector(javax.servlet.http.Cookie[] cookies, String domain) {
        List<Cookie> convertedCookies = new ArrayList<>();
        if (cookies != null) {
            for (javax.servlet.http.Cookie requestCookie : cookies) {
                BasicClientCookie cookie = new BasicClientCookie(requestCookie.getName(), requestCookie.getValue());
                cookie.setDomain(domain);
                cookie.setPath("/");
                cookie.setSecure(requestCookie.getSecure());
                cookie.setExpiryDate(new Date(System.currentTimeMillis() + MILLIS_ONE_WEEK));
                convertedCookies.add(cookie);
            }
        }
        return convertedCookies;
    }

}
