/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import java.util.Collections;
import java.util.List;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.endpoints.requests.MailsDeleteRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.MailsRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.MoveMailsRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.SetColorsRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.SetFlagsRequestBody;
import com.openexchange.mobile.api.facade.endpoints.responses.DeletedMailsResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.MailThreadsResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.MailsResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.MovedMailsResponseBody;
import com.openexchange.mobile.api.facade.endpoints.responses.ctos.MailCto;
import com.openexchange.mobile.api.facade.endpoints.responses.ctos.ThreadCto;
import com.openexchange.mobile.api.facade.models.DeletedMails;
import com.openexchange.mobile.api.facade.models.Mail;
import com.openexchange.mobile.api.facade.models.MailFlags;
import com.openexchange.mobile.api.facade.models.MailIds;
import com.openexchange.mobile.api.facade.models.MailThread;
import com.openexchange.mobile.api.facade.services.MailsService;
import com.openexchange.mobile.api.facade.utils.AttachmentUtil;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Path("/api-facade/v1/mails")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class MailsEndpoint {

    private static final String FIELD_COLOR = "color";

    private static final String FIELD_FOLDER_ID = "folderId";

    private static final String FIELD_TARGET_FOLDER_ID = "targetFolderId";

    private static final String FIELD_MAIL_ID = "mailId";

    private static final String FIELD_MAIL_IDS = "mailIds";

    private static final MapFunction<MailThread, ThreadCto> threadMapFunction = new MapFunction<MailThread, ThreadCto>() {
        @Override
        public ThreadCto map(MailThread thread) {
            return new ThreadCto(thread);
        }
    };

    private static final MapFunction<Mail, MailCto> mailMapFunction = new MapFunction<Mail, MailCto>() {
        @Override
        public MailCto map(Mail mail) {
            return new MailCto(mail);
        }
    };

    private final MailsService mailsService;

    @POST
    @Path("/stubs")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response getStubs(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Retrieving stubs");
        SessionData sessionData = RequestUtil.getSessionData(request);
        MailsRequestBody requestBody = JacksonUtil.parse(rawRequestBody, MailsRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        Boolean threadedValue = requestBody.getThreaded();
        boolean threaded = threadedValue == null || threadedValue.booleanValue();
        List<MailThread> threads = mailsService.getStubs(sessionData, requestBody.getFolderId(), threaded);
        List<ThreadCto> threadCtos = ListUtil.map(threads, threadMapFunction);
        MailThreadsResponseBody threadsResponseBody = new MailThreadsResponseBody(threadCtos);
        return RequestUtil.createCachableResponse(request, threadsResponseBody);
    }

    @POST
    @Path("/envelopes")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response getEnvelopes(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Retrieving envelopes");
        SessionData sessionData = RequestUtil.getSessionData(request);
        MailsRequestBody requestBody = JacksonUtil.parse(rawRequestBody, MailsRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        RequestUtil.checkParam(FIELD_MAIL_IDS, requestBody.getMailIds());
        List<Mail> mails = mailsService.getEnvelopes(sessionData, requestBody.getFolderId(), requestBody.getMailIds());
        List<MailCto> mailCtos = ListUtil.map(mails, mailMapFunction);
        MailsResponseBody mailResponseBody = new MailsResponseBody(mailCtos);
        return RequestUtil.jsonResponse(mailResponseBody);
    }

    @POST
    @Path("/complete")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response getMail(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Retrieving a complete mail");
        SessionData sessionData = RequestUtil.getSessionData(request);
        MailsRequestBody requestBody = JacksonUtil.parse(rawRequestBody, MailsRequestBody.class);
        String folderId = requestBody.getFolderId();
        RequestUtil.checkParam(FIELD_FOLDER_ID, folderId);
        RequestUtil.checkParam(FIELD_MAIL_ID, requestBody.getMailId());
        int mailTeasersDefaultLength = sessionData.getConfiguration().getMailTeasersDefaultLength();
        int maximumTeaserLength = requestBody.getMaximumTeaserLength() != null ? requestBody.getMaximumTeaserLength() : mailTeasersDefaultLength;
        Mail mail = mailsService.getMail(sessionData, folderId, requestBody.getMailId(), maximumTeaserLength);
        AttachmentUtil.initializeDownloadUrls(folderId, mail);
        MailCto mailCto = new MailCto(mail);
        MailsResponseBody mailsResponseBody = new MailsResponseBody(Collections.singletonList(mailCto));
        return RequestUtil.jsonResponse(mailsResponseBody);
    }

    @DELETE
    @Path("/delete")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response deleteMails(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Deleting mails");
        SessionData sessionData = RequestUtil.getSessionData(request);
        MailsDeleteRequestBody requestBody = JacksonUtil.parse(rawRequestBody, MailsDeleteRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        RequestUtil.checkParam(FIELD_MAIL_IDS, requestBody.getMailIds());
        DeletedMails deletedMails = mailsService.delete(sessionData, requestBody.getFolderId(), requestBody.getMailIds(), requestBody.isPurge());
        DeletedMailsResponseBody deletedMailsResponseBody = new DeletedMailsResponseBody(deletedMails);
        return RequestUtil.jsonResponse(deletedMailsResponseBody);
    }

    @POST
    @Path("/move")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response moveMails(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Moving mails");
        SessionData sessionData = RequestUtil.getSessionData(request);
        MoveMailsRequestBody requestBody = JacksonUtil.parse(rawRequestBody, MoveMailsRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        RequestUtil.checkParam(FIELD_MAIL_IDS, requestBody.getMailIds());
        RequestUtil.checkParam(FIELD_TARGET_FOLDER_ID, requestBody.getTargetFolderId());
        MailIds movedMails = mailsService.moveMails(sessionData, requestBody.getFolderId(), requestBody.getMailIds(), requestBody.getTargetFolderId());
        MovedMailsResponseBody movedMailsResponseBody = new MovedMailsResponseBody(movedMails.getMailIds());
        return RequestUtil.jsonResponse(movedMailsResponseBody);
    }

    @POST
    @Path("/flags/answered")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setAnsweredFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as answered");
        MailFlags flags = new MailFlags();
        flags.setAnswered(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/deleted")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setDeletedFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as deleted");
        MailFlags flags = new MailFlags();
        flags.setDeleted(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/draft")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setDraftFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as draft");
        MailFlags flags = new MailFlags();
        flags.setDraft(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/flagged")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setFlaggedFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as flagged");
        MailFlags flags = new MailFlags();
        flags.setFlagged(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/forwarded")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setForwardedFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as forwarded");
        MailFlags flags = new MailFlags();
        flags.setForwarded(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/recent")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setRecentFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as recent");
        MailFlags flags = new MailFlags();
        flags.setRecent(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/spam")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setSpamFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as spam");
        MailFlags flags = new MailFlags();
        flags.setSpam(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/unread")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setUnreadFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as unread");
        MailFlags flags = new MailFlags();
        flags.setSeen(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @POST
    @Path("/flags/user")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setUserFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Flagging as user");
        MailFlags flags = new MailFlags();
        flags.setUser(true);
        return setFlags(request, rawRequestBody, flags);
    }

    private Response setFlags(HttpServletRequest request, String rawRequestBody, MailFlags flags) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        SetFlagsRequestBody requestBody = JacksonUtil.parse(rawRequestBody, SetFlagsRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        RequestUtil.checkParam(FIELD_MAIL_IDS, requestBody.getMailIds());
        mailsService.setFlags(sessionData, requestBody.getFolderId(), flags, requestBody.getMailIds());
        return RequestUtil.emptyJsonResponse();
    }

    @DELETE
    @Path("/flags/answered")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearAnsweredFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as answered");
        MailFlags flags = new MailFlags();
        flags.setAnswered(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/deleted")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearDeletedFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as deleted");
        MailFlags flags = new MailFlags();
        flags.setDeleted(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/draft")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearDraftFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as draft");
        MailFlags flags = new MailFlags();
        flags.setDraft(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/flagged")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearFlaggedFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as flagged");
        MailFlags flags = new MailFlags();
        flags.setFlagged(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/forwarded")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearForwardedFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as forwarded");
        MailFlags flags = new MailFlags();
        flags.setForwarded(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/recent")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearRecentFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as recent");
        MailFlags flags = new MailFlags();
        flags.setRecent(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/spam")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearSpamFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as spam");
        MailFlags flags = new MailFlags();
        flags.setSpam(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/unread")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearUnreadFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as unread");
        MailFlags flags = new MailFlags();
        flags.setSeen(true);
        return setFlags(request, rawRequestBody, flags);
    }

    @DELETE
    @Path("/flags/user")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearUserFlags(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Clear flagging as user");
        MailFlags flags = new MailFlags();
        flags.setUser(true);
        return clearFlags(request, rawRequestBody, flags);
    }

    private Response clearFlags(HttpServletRequest request, String rawRequestBody, MailFlags flags) {
        SessionData sessionData = RequestUtil.getSessionData(request);
        SetFlagsRequestBody requestBody = JacksonUtil.parse(rawRequestBody, SetFlagsRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        RequestUtil.checkParam(FIELD_MAIL_IDS, requestBody.getMailIds());
        mailsService.clearFlags(sessionData, requestBody.getFolderId(), flags, requestBody.getMailIds());
        return RequestUtil.emptyJsonResponse();
    }

    @POST
    @Path("/color")
    @Consumes(MediaType.APPLICATION_JSON)
    public Response setColor(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Set colors");
        SessionData sessionData = RequestUtil.getSessionData(request);
        SetColorsRequestBody requestBody = JacksonUtil.parse(rawRequestBody, SetColorsRequestBody.class);
        RequestUtil.checkParam(FIELD_FOLDER_ID, requestBody.getFolderId());
        RequestUtil.checkParam(FIELD_MAIL_IDS, requestBody.getMailIds());
        RequestUtil.checkParam(FIELD_COLOR, requestBody.getColor());
        mailsService.setColors(sessionData, requestBody.getFolderId(), requestBody.getColor(), requestBody.getMailIds());
        return RequestUtil.emptyJsonResponse();
    }

}
