/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mailapp;

import java.util.LinkedList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.DefaultInterests;
import com.openexchange.config.Interests;
import com.openexchange.config.Reloadable;
import com.openexchange.java.Strings;

/**
 * {@link MailappConfig}
 *
 * @author <a href="mailto:kevin.ruthmann@open-xchange.com">Kevin Ruthmann</a>
 * @since v7.8.2
 */
public class MailappConfig implements Reloadable {

    private static final Logger LOG = LoggerFactory.getLogger(MailappConfig.class);
    private static final String CONFIG_FILE_NAME = "mailapp-config.yml";
    private static final String MAIL_APP_ENABLED = "mail_app_enabled";
    private Map<String, Object> configurations = null;

    @Override
    public void reloadConfiguration(ConfigurationService configService) {
        loadConfiguration(configService);
    }

    @SuppressWarnings("unchecked")
    public void loadConfiguration(ConfigurationService configService) {
        configurations = (Map<String, Object>) configService.getYaml(CONFIG_FILE_NAME);
    }

    @Override
    public Interests getInterests() {
        return DefaultInterests.builder().configFileNames(CONFIG_FILE_NAME).build();
    }

    @SuppressWarnings("unchecked")
    public boolean isEnabled(String hostName) {
        if (configurations == null) {
            return true;
        }
        LinkedList<Map<String, Object>> applicableConfigs = new LinkedList<>();
        for (Map.Entry<String, Object> configEntry : configurations.entrySet()) {
            Map<String, Object> possibleConfiguration = (Map<String, Object>) configEntry.getValue();
            if (looksApplicable(possibleConfiguration, hostName)) {
                // ensure that "all"-host-wildcards are applied first
                if ("all".equals(possibleConfiguration.get("host"))) {
                    applicableConfigs.addFirst(possibleConfiguration);
                } else {
                    applicableConfigs.add(possibleConfiguration);
                }
            } else {
                final String configName = configEntry.getKey();
                if (null == possibleConfiguration) {
                    LOG.debug("Empty configuration \"{}\" is not applicable", configName);
                } else {
                    LOG.debug("Configuration \"{}\" is not applicable: {}", configName, prettyPrint(configName, possibleConfiguration));
                }
            }
        }

        if (applicableConfigs.isEmpty()) {
            return true;
        }
        Map<String, Object> resultMap = applicableConfigs.getLast();
        if (!resultMap.containsKey(MAIL_APP_ENABLED)) {
            return true;
        }
        Object enabled = resultMap.get(MAIL_APP_ENABLED);
        if (!(enabled instanceof Boolean)) {
            return true;
        }

        return (Boolean) enabled;
    }

    String prettyPrint(final String configName, Map<String, Object> configuration) {
        if (null == configuration) {
            return "<not-set>";
        }

        final StringBuilder sb = new StringBuilder(configuration.size() << 4);
        final String indent = "    ";
        final String sep = Strings.getLineSeparator();

        sb.append(sep);
        prettyPrint(configName, configuration, indent, sep, sb);
        return sb.toString();
    }

    void prettyPrint(final String configName, Map<String, Object> configuration, final String indent, final String sep, final StringBuilder sb) {
        if (null != configuration) {
            sb.append(indent).append(configName).append(':').append(sep);

            for (final Entry<String, Object> entry : configuration.entrySet()) {
                final String key = entry.getKey();
                final Object value = entry.getValue();
                sb.append(indent).append(indent).append(key).append(": ");
                if (value instanceof String) {
                    sb.append('\'').append(value).append('\'');
                } else {
                    sb.append(value);
                }
                sb.append(sep);
            }
        }
    }

    private boolean looksApplicable(Map<String, Object> possibleConfiguration, String hostName) {
        if (possibleConfiguration == null) {
            return false;
        }

        // We need a hostname for the host and hostRegex check
        if (Strings.isNotEmpty(hostName)) {

            // Check "host"
            {
                final String host = (String) possibleConfiguration.get("host");
                if (host != null) {
                    if ("all".equals(host)) {
                        return true;
                    }

                    if (host.equals(hostName)) {
                        return true;
                    }

                    // Not applicable according to host check
                    LOG.debug("Host '{}' does not apply to {}", host, hostName);
                }
            }

            // Check "hostRegex"
            {
                final String keyHostRegex = "hostRegex";
                final String hostRegex = (String) possibleConfiguration.get(keyHostRegex);
                if (hostRegex != null) {
                    try {
                        final Pattern pattern = Pattern.compile(hostRegex);

                        if (pattern.matcher(hostName).find()) {
                            return true;
                        }

                        // Not applicable according to hostRegex check
                        LOG.debug("Host-Regex '{}' does not match {}", hostRegex, hostName);
                    } catch (final PatternSyntaxException e) {
                        // Ignore. Treat as absent.
                        LOG.debug("Invalid regex pattern for {}: {}", keyHostRegex, hostRegex, e);
                    }
                }
            }
        }

        return false;
    }

}
