/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobilepush.events.storage.rdb;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import com.openexchange.context.ContextService;
import com.openexchange.database.DatabaseService;
import com.openexchange.exception.OXException;
import com.openexchange.mobilepush.MobilePushExceptionCodes;
import com.openexchange.mobilepush.MobilePushProvider;
import com.openexchange.mobilepush.events.storage.MobilePushStorageService;
import com.openexchange.mobilepush.events.storage.Subscription;
import com.openexchange.mobilepush.events.storage.osgi.Services;
import com.openexchange.session.Session;
import com.openexchange.tools.sql.DBUtils;

/**
 * {@link RdbMobilePushStorageImpl}
 *
 * @author <a href="mailto:lars.hoogestraat@open-xchange.com">Lars Hoogestraat</a>
 */
public class RdbMobilePushStorageImpl implements MobilePushStorageService {

    private final DatabaseService databaseService;

    /**
     * Initializes a new {@link RdbSubscriptionStore}.
     *
     * @throws OXException If the database service is missing
     */
    public RdbMobilePushStorageImpl() throws OXException {
        super();
        this.databaseService = Services.getService(DatabaseService.class);
    }

    @Override
    public Subscription createSubscription(Session session, String token, String serviceId, MobilePushProvider providerId) throws OXException {
        Subscription subscription = new Subscription(
            session.getContextId(), session.getUserId(), token, serviceId, providerId, System.currentTimeMillis());
        Connection connection = databaseService.getWritable(session.getContextId());
        try {
            if (0 == replaceSubscription(connection, session, subscription)) {
                throw MobilePushExceptionCodes.DB_ERROR.create("Subscription not added: " + subscription);
            }
        } catch (SQLException e) {
            throw MobilePushExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            databaseService.backWritable(session.getContextId(), connection);
        }
        return subscription;
    }

    private static int replaceSubscription(Connection connection, Session session, Subscription subscription) throws SQLException, OXException {
        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(Statements.REPLACE_OR_ADD_SUBSCRIPTION);
            stmt.setInt(1, subscription.getContextId());
            stmt.setString(2, subscription.getServiceId());
            stmt.setString(3, subscription.getToken());
            stmt.setString(4, subscription.getProviderName());
            stmt.setInt(5, subscription.getUserId());
            stmt.setLong(6, subscription.getTimestamp());
            return Statements.logExecuteUpdate(stmt);
        } finally {
            DBUtils.closeSQLStuff(stmt);
        }
    }

    @Override
    public boolean updateToken(int contextId, String token, String serviceId, String newToken) throws OXException {
        Connection connection = databaseService.getWritable(contextId);
        try {
            // Delete existing new tokens first if any
            deleteSubscription(contextId, newToken, serviceId);

            return 0 < updateSubscriptions(connection, contextId, serviceId, token, newToken);
        } catch (SQLException e) {
            throw MobilePushExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            databaseService.backWritable(contextId, connection);
        }
    }

    private static int updateSubscriptions(Connection connection, int contextId, String serviceId, String token, String newToken) throws SQLException, OXException {
        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(Statements.UPDATE_TOKENS);
            //UPDATE ... SET:
            stmt.setString(1, newToken);
            stmt.setLong(2, System.currentTimeMillis());
            //WHERE:
            stmt.setInt(3, contextId);
            stmt.setString(4, serviceId);
            stmt.setString(5, token);
            return Statements.logExecuteUpdate(stmt);
        } finally {
            DBUtils.closeSQLStuff(stmt);
        }
    }

    @Override
    public boolean deleteSubscription(int contextId, String token, String serviceId, MobilePushProvider providerId) throws OXException {
        Connection connection = databaseService.getWritable(contextId);
        try {
            return 0 < deleteSubscription(connection, contextId, token, serviceId, providerId);
        } catch (SQLException e) {
            throw MobilePushExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            databaseService.backWritable(contextId, connection);
        }
    }

    private static int deleteSubscription(Connection connection, int contextId, String token, String serviceId, MobilePushProvider provider) throws SQLException, OXException {
        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(Statements.DELETE_TOKEN_BY_PROVIDER);
            stmt.setInt(1, contextId);
            stmt.setString(2, serviceId);
            stmt.setString(3, provider.getProviderName());
            stmt.setString(4, token);
            return Statements.logExecuteUpdate(stmt);
        } finally {
            DBUtils.closeSQLStuff(stmt);
        }
    }

    @Override
    public boolean deleteSubscription(int contextId, String token, String serviceId) throws OXException {
        Connection connection = databaseService.getWritable(contextId);
        try {
            return 0 < deleteSubscription(connection, contextId, token, serviceId);
        } catch (SQLException e) {
            throw MobilePushExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            databaseService.backWritable(contextId, connection);
        }
    }

    @Override
    public int deleteSubscription(String token, String serviceId) throws OXException {
        int removed = 0;
        ContextService contextService = Services.getService(ContextService.class);
        Set<Integer> allContextIDs = new HashSet<Integer>(contextService.getAllContextIds());
        while (false == allContextIDs.isEmpty()) {
            /*
             * Delete for whole schema using connection for first context
             */
            int contextId = allContextIDs.iterator().next().intValue();
            Connection connection = databaseService.getWritable(contextId);
            try {
                removed += deleteSubscription(connection, contextId, token, serviceId);
            } catch (SQLException e) {
                throw MobilePushExceptionCodes.SQL_ERROR.create(e, e.getMessage());
            } finally {
                databaseService.backWritable(contextId, connection);
            }
            /*
             * Remember processed contexts
             */
            int[] contextsInSameSchema = databaseService.getContextsInSameSchema(contextId);
            for (int cid : contextsInSameSchema) {
                allContextIDs.remove(Integer.valueOf(cid));
            }
        }
        return removed;
    }

    private static int deleteSubscription(Connection connection, int contextId, String token, String serviceId) throws SQLException {
        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(Statements.DELETE_TOKEN_BY_SERVICE_ID);
            stmt.setInt(1, contextId);
            stmt.setString(2, serviceId);
            stmt.setString(3, token);
            return Statements.logExecuteUpdate(stmt);
        } finally {
            DBUtils.closeSQLStuff(stmt);
        }
    }

    @Override
    public List<Subscription> getSubscriptions(int contextId, int userId, String serviceId, MobilePushProvider provider) throws OXException {
        Connection connection = databaseService.getReadOnly(contextId);
        try {
            return selectSubscriptions(connection, contextId, userId, serviceId, provider.getProviderName());
        } catch (SQLException e) {
            throw MobilePushExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            databaseService.backReadOnly(contextId, connection);
        }
    }

    private static List<Subscription> selectSubscriptions(Connection connection, int contextId, int userId, String service, String providerName) throws SQLException {
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            int index = 0;
            stmt = connection.prepareStatement(Statements.SELECT_SUBSCRIPTIONS);
            stmt.setInt(++index, contextId);
            stmt.setInt(++index, userId);
            stmt.setString(++index, service);
            stmt.setString(++index, providerName);
            rs = Statements.logExecuteQuery(stmt);
            if (false == rs.next()) {
                return Collections.emptyList();
            }

            List<Subscription> subscriptions = new LinkedList<Subscription>();
            do {
                int cid = rs.getInt(1);
                String resService = rs.getString(2);
                String resToken = rs.getString(3);
                String resProvider = rs.getString(4);
                int resUserId = rs.getInt(5);
                long resTimestamp = rs.getLong(6);
                subscriptions.add(new Subscription(cid, resUserId, resToken, resService, MobilePushProvider.parseProvider(resProvider), resTimestamp));
            } while (rs.next());
            return subscriptions;
        } finally {
            DBUtils.closeSQLStuff(rs, stmt);
        }
    }
}
