/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.fileitem.impl;

import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.apache.commons.collections.BidiMap;
import org.apache.commons.lang.ArrayUtils;
import org.json.JSONObject;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.FileStorage;
import com.openexchange.imageconverter.api.AccessOption;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItem;
import com.openexchange.imageconverter.api.IFileItemReadAccess;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.IFileItemWriteAccess;
import com.openexchange.imageconverter.api.ISubGroup;

/**
 * {@link FileItemService}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
/**
 * {@link FileItemService}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemService implements IFileItemService {

    // - Default ids for group and subgroup ------------------------------------

    final public static String GROUP_ID_DEFAULT = "764147e8-ee66-412b-ad88-9ea23283a866";

    final public static String SUBGROUP_ID_DEFAULT = "fc076f7f-3163-4d12-9be7-4037a4fa62eb";

    /**
     * Initializes a new {@link FileItemService}.
     */
    public FileItemService(final Optional<BidiMap> fileStoreMapOpt, final Optional<FileItemDatabase> databaseOpt) {

        super();

        // check for valid FileStores and database
        m_fileStorageMap = fileStoreMapOpt.isPresent() ? fileStoreMapOpt.get() : null;
        m_database = databaseOpt.isPresent() ? databaseOpt.get() : null;

        if ((null != m_fileStorageMap) && (m_fileStorageMap.size() > 0) && (null != m_database)) {
            final int fileStoreCount =m_fileStorageMap.size();
            int curFileStoreArrayPos = -1;

            m_fileStorageArray = new FileStorage[fileStoreCount];

            for (final Object curFileStorage : m_fileStorageMap.keySet()) {
                m_fileStorageArray[++curFileStoreArrayPos] = (FileStorage) curFileStorage;
            }

            (m_fileStoreRemover = new FileStoreRemover(this)).start();
            m_valid = true;
        } else {
            m_valid = false;
        }
    }

    // - IFileItemService ------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#isValid()
     */
    @Override
    public boolean isValid() {
        return m_valid;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getDBType()
     */
    @Override
    public DatabaseType getDatabaseType() {
        return hasDatabase() ?
            m_database.getDatabaseType() :
                DatabaseType.MYSQL;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#registerGroup(java.lang.String, java.lang.String[])
     */
    @Override
    public void registerGroup(final String groupId, final String... keyN) throws FileItemException {
        if (isEmpty(groupId)) {
            throw new FileItemException("FileItemService not able to register group with empty groupId");
        }

        if (hasDatabase()) {
            m_database.registerGroup(groupId, keyN);
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getCustomKeys(java.lang.String)
     */
    @Override
    public String[] getCustomKeys(final String groupId) throws FileItemException {
        if (isEmpty(groupId)) {
            throw new FileItemException("Invalid groupId");
        }

        if (hasDatabase()) {
            final Set<String> keySet = m_database.getCustomPropertyKeys(groupId);
            return keySet.toArray(new String[keySet.size()]);
        }

        return ArrayUtils.EMPTY_STRING_ARRAY;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#hasCustomKey(java.lang.String, java.lang.String)
     */
    @Override
    public boolean hasCustomKey(final String groupId, final String key) throws FileItemException {
        if (isEmpty(groupId)) {
            throw new FileItemException("Invalid groupId");
        } else if (isEmpty(key)) {
            throw new FileItemException("Invalid key");
        }

        return hasDatabase() ?
            m_database.hasCustomPropertyKey(groupId, key) :
                false;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getUserData(java.lang.String)
     */
    @Override
    public JSONObject getUserData(@NonNull final String groupId) throws FileItemException {
        if (isEmpty(groupId)) {
            throw new FileItemException("Invalid groupId");
        }

        return hasDatabase() ?
            m_database.getUserData(groupId) :
                new JSONObject();
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#setUserData(java.lang.String, org.json.JSONObject)
     */
    @Override
    public boolean setUserData(final String groupId, final JSONObject userData) throws FileItemException {
        if (isEmpty(groupId)) {
            throw new FileItemException("Invalid groupId");
        }

        return (hasDatabase() && (null != userData)) ?
            m_database.setUserData(groupId, userData) :
                false;
    }

    // -------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#containsGroup(java.lang.String)
     */
    @Override
    public boolean containsGroup(final String groupId) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId)) ?
            m_database.contains(groupId) :
                false;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#containsSubGroup(java.lang.String, java.lang.String)
     */
    @Override
    public boolean containsSubGroup(final String groupId, final String subGroupId) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId) && FileItemUtils.isValid(subGroupId)) ?
            m_database.contains(groupId, subGroupId) :
                false;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#containsSubGroup(java.lang.String, java.lang.String)
     */
    @Override
    public boolean contains(final String groupId, final String subGroupId, final String fileId) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId) && FileItemUtils.isValid(subGroupId) && FileItemUtils.isValid(fileId)) ?
            m_database.contains(groupId, subGroupId) :
                false;
    }

    // -------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#get(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public IFileItem get(String groupId, String subGroupId, String fileId) throws FileItemException {
        FileItem ret = implGetCheckedFileItem(implCreateFileItem(groupId, subGroupId, fileId));

        if (hasDatabase() && (null != ret)) {
            final FileStoreData fileStoreData = m_database.getFileStoreData(ret);

            if (FileItemUtils.isValid(fileStoreData)) {
                ret.setFileStoreData(fileStoreData);
            } else {
                ret = null;
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#get(java.lang.String, java.lang.String)
     */
    @Override
    public IFileItem[] get(final String groupId, final String subGroupId) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId) && FileItemUtils.isValid(subGroupId)) ?
            m_database.getFileItems(groupId, subGroupId) :
                new FileItem[0];
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#get(java.lang.String, java.util.Properties)
     */
    @Override
    public IFileItem[] get(final String groupId, final Properties properties) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId) && (null != properties) && (properties.size() > 0)) ?
            m_database.getFileItems(groupId, properties) :
                new FileItem[0];
    }

    // -------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getSubGroupCount(java.lang.String)
     */
    @Override
    public long getSubGroupCount(final String groupId) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId)) ?
            m_database.getSubGroupCount(groupId) :
                0;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getSubGroups(java.lang.String)
     */
    @Override
    public String[] getSubGroups(final String groupId) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId)) ?
            m_database.getSubGroupIds(groupId) :
                ArrayUtils.EMPTY_STRING_ARRAY;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getSubGroupsBy(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public String[] getSubGroupsBy(final String groupId, @Nullable final String whereClause, @Nullable String limitClause) throws FileItemException {
        return (hasDatabase() && isNotEmpty(whereClause) || isNotEmpty(limitClause)) ?
            m_database.getSubGroupIdsBy(groupId, whereClause, limitClause) :
                ArrayUtils.EMPTY_STRING_ARRAY;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getSubGroupsBy(java.lang.String, java.lang.String, java.lang.String, java.util.function.Consumer)
     */
    @Override
    public long getSubGroupsBy(String groupId, String whereClause, String limitClause, Consumer<ISubGroup> subGroupConsumer) throws FileItemException {
        return (hasDatabase() && FileItemUtils.isValid(groupId)) ?
            m_database.getSubGroupsBy(groupId, whereClause, limitClause, subGroupConsumer) :
                0;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getGroupCount()
     */
    @Override
    public long getGroupCount() throws FileItemException {
        return hasDatabase() ?
            m_database.getGroupCount() :
                0;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getGroups()
     */
    @Override
    public String[] getGroups() throws FileItemException {
        return hasDatabase() ?
            m_database.getGroupIds() :
                ArrayUtils.EMPTY_STRING_ARRAY;
    }

    // -------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getGroupLength(java.lang.String)
     */
    @Override
    public long getGroupLength(String groupId) throws FileItemException {
        return hasDatabase() ?
            m_database.getGroupLength(groupId) :
                0;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getGroupLength(java.lang.String, java.util.Properties)
     */
    @Override
    public long getGroupLength(String groupId, Properties properties) throws FileItemException {
        return hasDatabase() ?
            m_database.getGroupLength(groupId, null, properties) :
                0;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getSubGroupLength(java.lang.String, java.lang.String)
     */
    @Override
    public long getSubGroupLength(String groupId, String subGroupId) throws FileItemException {
        return hasDatabase() ?
            m_database.getGroupLength(groupId, subGroupId, null) :
                0;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getSubGroupLength(java.lang.String, java.lang.String, java.util.Properties)
     */
    @Override
    public long getSubGroupLength(String groupId, String subGroupId, Properties properties) throws FileItemException {
        return hasDatabase() ?
            m_database.getGroupLength(groupId, subGroupId, properties) :
                0;
    }

    // -------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#remove(com.openexchange.imageconverter.api.IFileItem)
     */
    @Override
    public boolean remove(IFileItem iFileItem) throws FileItemException {
        final FileItem fileItem = implGetCheckedFileItem(iFileItem);
        boolean ret = false;

        if (hasDatabase() && (null != fileItem)) {
            final String fileItemKey = fileItem.getKey();

            try {
                final List<FileStoreData> deletedFileStoreDataList = new ArrayList<>();

                ElementLocker.lock(fileItemKey);

                if (1 == m_database.deleteEntry(fileItem.getGroupId(), fileItem.getSubGroupId(), fileItem.getFileId(), deletedFileStoreDataList)) {
                    ret = (m_fileStoreRemover.addFileStoreDatasToRemove(deletedFileStoreDataList) > 0);
                }
            } finally {
                ElementLocker.unlock(fileItemKey);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#remove(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public boolean remove(String groupId, String subGroupId, String fileId) throws FileItemException {
        return remove(implCreateFileItem(groupId, subGroupId, fileId));
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#remove(com.openexchange.imageconverter.api.IFileItem[])
     */
    @Override
    public int remove(IFileItem[] fileElements) throws FileItemException {
        int removedFileCount = 0;

        if (null != fileElements) {
            for (final IFileItem curFileItem : fileElements) {
                    remove(curFileItem);
                    ++removedFileCount;
            }
        }

        return removedFileCount;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#remove(java.lang.String, java.util.Properties)
     */
    @Override
    public int remove(final String groupId, final Properties properties) throws FileItemException {
        int ret = 0;

        if (hasDatabase() && isNotEmpty(groupId) && (null != properties) && (properties.size() > 0)) {
            final List<FileStoreData> deletedFileStoreDataList = new ArrayList<>(8192);
            final int deletedEntries = m_database.deleteByProperties(groupId, properties, deletedFileStoreDataList);

            if (deletedEntries > 0)  {
                ret = m_fileStoreRemover.addFileStoreDatasToRemove(deletedFileStoreDataList);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#removeSubGroup(java.lang.String, java.lang.String)
     */
    @Override
    public int removeSubGroup(final String groupId, final String subGroupId) throws FileItemException {
        int ret = 0;

        if (hasDatabase() && FileItemUtils.isValid(groupId) && FileItemUtils.isValid(subGroupId)) {
            final List<FileStoreData> deletedFileStoreDataList = new ArrayList<>(8192);
            final int deletedEntries = m_database.deleteByGroupSubgroup(groupId, subGroupId, deletedFileStoreDataList);

            if (deletedEntries > 0)  {
                ret = m_fileStoreRemover.addFileStoreDatasToRemove(deletedFileStoreDataList);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#removeSubGroups(java.lang.String, java.lang.String[])
     */
    @Override
    public int removeSubGroups(final String groupId, final String[] subGroupIds) throws FileItemException {
        int ret = 0;

        if (hasDatabase() && FileItemUtils.isValid(groupId) && FileItemUtils.isValid(subGroupIds)) {
            final List<FileStoreData> deletedFileStoreDataList = new ArrayList<>(8192);
            final String[] targetSubGroupIds = Arrays.asList(subGroupIds).stream().
                                                   filter(FileItemUtils::isValid).
                                                   toArray(p -> new String[p]);

            final int deletedEntries = m_database.deleteByGroupSubgroups(groupId, targetSubGroupIds, deletedFileStoreDataList);

            if (deletedEntries > 0)  {
                ret = m_fileStoreRemover.addFileStoreDatasToRemove(deletedFileStoreDataList);
            }
        }

        return ret;
    }


    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#removeGroup(java.lang.String)
     */
    @Override
    public int removeGroup(final String groupId) throws FileItemException {
        int ret = 0;

        if (hasDatabase() && FileItemUtils.isValid(groupId)) {
            final List<FileStoreData> deletedFileStoreDataList = new ArrayList<>(8192);
            final int deletedEntries = m_database.deleteByGroupSubgroup(groupId, null, deletedFileStoreDataList);

            if (deletedEntries > 0)  {
                ret = m_fileStoreRemover.addFileStoreDatasToRemove(deletedFileStoreDataList);
            }
        }

        return ret;
    }

    // -------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getReadAccess(com.openexchange.imageconverter.api.IFileItem, com.openexchange.imageconverter.api.AccessOption[])
     */
    @Override
    public IFileItemReadAccess getReadAccess(final IFileItem iFileItem, AccessOption... accessOption) throws FileItemException {
        final FileItem fileItem = implGetCheckedFileItem(iFileItem);
        FileItemReadAccess ret = null;

        if (null != fileItem) {
            final String fileItemKey = fileItem.getKey();

            try {
                ElementLocker.lock(fileItemKey);

                ret = new FileItemReadAccess(this, m_database, fileItem, accessOption);

                try {
                    ret.open();
                } catch (Exception e) {
                    FileItemUtils.close(ret);
                    ret = null;

                    throw new FileItemException(e);
                }
            } finally {
                ElementLocker.unlock(fileItemKey);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getReadAccess(java.lang.String, java.lang.String, java.lang.String, com.openexchange.imageconverter.api.AccessOption[])
     */
    @Override
    public IFileItemReadAccess getReadAccess(String groupId, String subGroupId, String fileId, AccessOption... accessOptions) throws FileItemException {
        return getReadAccess(implCreateFileItem(groupId, subGroupId, fileId), accessOptions);
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getWriteAccess(com.openexchange.imageconverter.api.IFileItem, com.openexchange.imageconverter.api.AccessOption[])
     */
    @Override
    public IFileItemWriteAccess getWriteAccess(final IFileItem ifileItem, AccessOption... accessOption) throws FileItemException {
        final FileItem fileItem = implGetCheckedFileItem(ifileItem);
        FileItemWriteAccess ret = null;

        if (null != fileItem) {
            final String fileItemKey = fileItem.getKey();

            try {
                ElementLocker.lock(fileItemKey);

                ret = new FileItemWriteAccess(this, m_database, fileItem, accessOption);

                try {
                    ret.open();
                } catch (Exception e) {
                    FileItemUtils.close(ret);
                    ret = null;

                    throw new FileItemException(e);
                }
            } finally {
                ElementLocker.unlock(fileItemKey);
            }
       }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemService#getWriteAccess(java.lang.String, java.lang.String, java.lang.String, com.openexchange.imageconverter.api.AccessOption[])
     */
    @Override
    public IFileItemWriteAccess getWriteAccess(String groupId, String subGroupId, String fileId, AccessOption...accessOptions) throws FileItemException {
        return getWriteAccess(implCreateFileItem(groupId, subGroupId, fileId), accessOptions);
    }

    // - Package internal API --------------------------------------------------

    /**
     * @return
     */
    public boolean hasDatabase() {
        return (null != m_database);
    }

    /**
     *
     */
    public void shutdown() {
        if (m_running.compareAndSet(true, false)) {
            if (null != m_fileStoreRemover) {
                m_fileStoreRemover.shutdown();
            }

            FileItemUtils.shutdown();
        }
    }

    /**
     * @param fileStoreData
     * @return
     */
    protected FileStorage getFileStorage(@NonNull FileStoreData fileStoreData) {
        return (FileStorage) m_fileStorageMap.getKey(Integer.valueOf(fileStoreData.getFileStoreNumber()));
    }

    /**
     * @param fileItem
     * @return
     * @throws OXException
     */
    protected InputStream getFileStoreInputStream(@NonNull FileItem fileItem) throws OXException {
        InputStream ret = null;

        if (FileItemUtils.isValid(fileItem)) {
            final FileStoreData fileStoreData = fileItem.getFileStoreData();

            if (FileItemUtils.isValid(fileStoreData)) {
                final FileStorage fileStorage = getFileStorage(fileStoreData);

                if (null != fileStorage) {
                    ret = fileStorage.getFile(fileStoreData.getFileStoreId());
                }
            }
        }

        return ret;
    }

    /**
     * @param fileItem
     * @return
     */
    protected FileStoreData createNewFileStoreFile(@NonNull IFileItem fileItem,
        @NonNull FileItemProperties fileItemProperties, InputStream fileContent) throws IOException {

        final int fileStoreCount = m_fileStorageArray.length;
        FileStoreData ret = null;

        if (fileStoreCount > 0) {
            // walk over all available FileStores round robin wise
            // and create a new FileStore file
            for (int tryNumber = 0; (null == ret) && (tryNumber < fileStoreCount); ++tryNumber) {
                final FileStorage fileStore = m_fileStorageArray[implGetNextFileStorePos()];

                try {
                    // create new FileStore file
                    final String fileStoreId = fileStore.saveNewFile(fileContent);

                    if (FileItemUtils.isValid(fileStoreId)) {
                        ret = new FileStoreData(((Integer) m_fileStorageMap.get(fileStore)).intValue(), fileStoreId);

                        fileItemProperties.setLength(fileStore.getFileSize(fileStoreId));
                        fileItemProperties.setModificationDateMillis(System.currentTimeMillis());
                    }
                } catch (OXException e) {
                    throw new IOException("Creation of new FileStore file failed: " + fileItem.toString(), e);
                }
            }
        }

        return ret;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     */
    protected synchronized int implGetNextFileStorePos() {
        return (m_curFileStorePos = ++m_curFileStorePos % m_fileStorageArray.length);
    }

    /**
     * @param groupId
     * @param subGroupId
     * @param fileId
     * @return
     */
    protected static IFileItem implCreateFileItem(String groupId, String subGroupId, String fileId) {
        return new FileItem(groupId, subGroupId, fileId);
    }

    /**
     * @param fileItem
     * @return
     * @throws FileItemException
     */
    protected static FileItem implGetCheckedFileItem(IFileItem fileItem) throws FileItemException {
        if (!(fileItem instanceof FileItem)) {
            throw new FileItemException("Unsupported implementation of IFileItem used");
        } else if (FileItemUtils.isInvalid((FileItem) fileItem)) {
            throw new FileItemException("GrouId, SubGroupId and FileId must be set to valid ids");
        }

        return (FileItem) fileItem;
    }

    // - Members ---------------------------------------------------------------

    final protected boolean m_valid;

    protected FileItemDatabase m_database = null;

    protected BidiMap m_fileStorageMap = null;

    protected FileStorage[] m_fileStorageArray = new FileStorage[0];

    protected FileStoreRemover m_fileStoreRemover = null;

    protected int m_curFileStorePos = -1;

    protected AtomicBoolean m_running = new AtomicBoolean(true);
}
