/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.fileitem.impl;

import static com.openexchange.fileitem.impl.FileItemUtils.LOG;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;
import org.apache.commons.io.FileUtils;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.AccessOption;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemWriteAccess;


/**
 * {@link FileItemWriteAccess}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemWriteAccess extends FileItemReadAccess implements IFileItemWriteAccess  {

    final private static InputStream EMPTY_INPUT_STREAM = new ByteArrayInputStream(new byte[] {});

    /**
     * Initializes a new {@link FileItemWriteAccess}.
     * @param fileItem
     * @param fileItem
     * @throws FileItemException
     */
    protected FileItemWriteAccess(@NonNull FileItemService fileItemService,
        @NonNull final FileItemDatabase database,
        @NonNull final FileItem fileItem,
        AccessOption... accessOption) {

        super(fileItemService, database, fileItem, accessOption);
    }

    /* (non-Javadoc)
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() throws IOException {
        File outputFile = null;

        try {
            final FileStoreData newFileStoreData = createNewFileStoreFile();

            if ((null != newFileStoreData) && (null != (outputFile = getOutputFile()))) {
                m_fileItemProperties.setLength(outputFile.length());
                m_fileItemProperties.setModificationDateMillis(outputFile.lastModified());

                if (m_updateExistingFileItem) {
                    // remove old fileItem entry as well as old FileStore file in update
                    // case first since we cannot add new entries with same PRIMARY keys twice
                    final FileStoreData oldFileStoreData = m_fileItem.getFileStoreData();

                    try {
                        // delete old entry in case of an update
                        final List<FileStoreData> deletedFileStoreDataList = new ArrayList<>();
                        m_database.deleteByFileStoreData(m_fileItem.getGroupId(), oldFileStoreData, deletedFileStoreDataList);
                    } catch (FileItemException e) {
                        LOG.warn("Unable to remove old FileStoreData from database after updating fileItem: {}", Throwables.getRootCause(e).getMessage());
                        throw new IOException("Error while updating database entry", e);
                    } finally {
                        try {
                            FileItemUtils.deleteFileStoreFile(m_fileItemService, oldFileStoreData);
                        } catch (FileItemException e) {
                            LOG.warn("Unable to remove FileStore file after error occured: {}", Throwables.getRootCause(e).getMessage());
                            // ok, there's no more, we can do
                        }
                    }
                }

                m_fileItem.setFileStoreData(newFileStoreData);

                try {
                    m_database.createEntry(m_fileItem, newFileStoreData, m_fileItemProperties);
                } catch (FileItemException e) {
                    try {
                        // delete FileStore file in case of Database error
                        FileItemUtils.deleteFileStoreFile(m_fileItemService, newFileStoreData);
                    } catch (FileItemException e1) {
                        LOG.warn("Unable to remove FileStore file after error occured: {}", Throwables.getRootCause(e1).getMessage());
                        // ok, there's no more, we can do
                    } finally {
                        // reset FileStoreData in case of failure in every case
                        m_fileItem.setFileStoreData(null);
                    }

                    throw new IOException("Error while creating database entry", e);
                }
            }
        } finally {
            // close output stream
            FileItemUtils.close(m_fileContentOutputStm);
            m_fileContentOutputStm = null;

            // get rid of temp. output file
            FileUtils.deleteQuietly(m_fileContentOutputFile);
            m_fileContentOutputFile = null;

            FileUtils.deleteQuietly(outputFile);

            try {
                // calling close method from superclass to cleanup object.
                // m_fileItem and m_fileItemProperties are not set to null
                // to be still accessible after closing the item access
                super.close();
            } finally {
                // unlock access time (from open to close) lock, that was
                // previously acquired in FileItemWriteAccess#open method
                ElementLocker.unlock(m_fileItemKey);
           }
        }
    }

    // - IFileItemWriteAccess ------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemWriteAccess#getOutputStream()
     */
    @Override
    public OutputStream getOutputStream() {
        return m_fileContentOutputStm;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemWriteAccess#getOutputFile()
     */
    @Override
    public File getOutputFile() throws IOException {
        if (null == m_fileContentOutputFile) {
            if (null != m_fileContentOutputStm) {
                if (null != (m_fileContentOutputFile = FileItemUtils.createTempFile())) {
                    FileUtils.writeByteArrayToFile(m_fileContentOutputFile, ((ByteArrayOutputStream) m_fileContentOutputStm).toByteArray());

                    // the previously used output stream is closed and not valid anymore
                    FileItemUtils.close(m_fileContentOutputStm);
                    m_fileContentOutputStm = null;
                }
            } else {
                FileUtils.deleteQuietly(m_fileContentOutputFile);
                m_fileContentOutputFile = null;
            }
        }

        return m_fileContentOutputFile;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemWriteAccess#setKeyValue(java.lang.String, java.lang.String)
     */
    @Override
    public void setKeyValue(final String key, final String value) {
        if (null != m_fileItemProperties) {
            m_fileItemProperties.setCustomKeyValue(key, value);
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     * @throws FileItemException
     * @throws NoSuchElementException
     * @throws IOException
     */
    @Override
    protected void open() throws FileItemException, IOException {
        // Global lock for the lifetime of this access object => will finally be unlocked in #close call
        // As such, an open call must be followed by a close call in order to ensure correct unlocking
        ElementLocker.lock(m_fileItemKey);

        if (!tryOpen()) {
            // no file existing, it will be created by us in #close method;
            // in order to work this way, we need exclusive file access locking
            m_updateExistingFileItem = false;
            m_fileItemProperties = new FileItemProperties(m_database.getCustomPropertyKeys(m_fileItem.getGroupId()), System.currentTimeMillis());
            m_fileContentInputStm = EMPTY_INPUT_STREAM;
        }

        // create output file/stream objects to work with
        if (null != (m_fileContentOutputFile = FileItemUtils.createTempFile())) {
            m_fileContentOutputStm = FileUtils.openOutputStream(m_fileContentOutputFile);
        }
    }

    /**
     * @return
     * @throws IOException
     */
    protected FileStoreData createNewFileStoreFile() throws IOException {
        FileStoreData ret = null;
        boolean writeOutput = (null != m_fileContentOutputFile);

        if (null != m_fileContentOutputStm) {
            m_fileContentOutputStm.flush();
            m_fileContentOutputStm.close();
            writeOutput = true;
        }

        if (writeOutput) {
            try (final InputStream newContentInputStm = (null != m_fileContentOutputFile) ?
                new BufferedInputStream(FileUtils.openInputStream(m_fileContentOutputFile)) :
                    new ByteArrayInputStream(((ByteArrayOutputStream) m_fileContentOutputStm).toByteArray())) {

                ret = m_fileItemService.createNewFileStoreFile(m_fileItem, m_fileItemProperties, newContentInputStm);
            }
        }

        return ret;
    }

    // - Members ---------------------------------------------------------------

    protected OutputStream m_fileContentOutputStm = null;

    protected File m_fileContentOutputFile = null;

    protected boolean m_updateExistingFileItem = true;
 }
