/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.impl;

import java.util.Properties;
import com.openexchange.annotation.NonNull;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.pool.HikariPool;

/**
 * {@link HikariSetup}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
class HikariSetup {

    final public static int DEFAULT_POOL_MAXSIZE = 30;
    final public static long DEFAULT_POOL_CONNECTTIMEOUTMILLIS = 120000;
    final public static long DEFAULT_POOL_IDLETIMEOUTMILLIS = 300000;

    /**
     * Initializes a new {@link HikariSetup}.
     */
    @SuppressWarnings("unused")
    private HikariSetup() {
        super();
    }

    /**
     * Initializes a new {@link HikariSetup}.
     */
    HikariSetup(@NonNull final String driverClassName,
                @NonNull final String jdbcUrl,
                @NonNull final String userName,
                @NonNull final String password,
                @NonNull final Properties dbProperties,
                final int poolMaxSize,
                final long poolConnectTimeoutMillis,
                final long poolIdleTimeoutMillis) {
        super();

        m_driverClassName = driverClassName;
        m_jdbcUrl = jdbcUrl;
        m_username = userName;
        m_password = password;
        m_dbProperties = dbProperties;

        m_poolMaxSize = poolMaxSize;
        m_poolConnectTimeoutMillis = poolConnectTimeoutMillis;
        m_poolIdleTimeoutMillis = poolIdleTimeoutMillis;
    }

    /**
     * @param readOnly
     * @return
     */
    HikariPool getConnectionPool(final boolean readOnly) {
        final HikariConfig poolConfig = new HikariConfig();

        // DB related Hikari properties
        poolConfig.setDriverClassName(m_driverClassName);
        poolConfig.setJdbcUrl(m_jdbcUrl);
        poolConfig.setUsername(m_username);
        poolConfig.setPassword(m_password);
        poolConfig.setAutoCommit(false);
        poolConfig.setReadOnly(readOnly);

        // common DB properties
        for (final String curPropKey : m_dbProperties.stringPropertyNames()) {
            poolConfig.addDataSourceProperty(curPropKey, m_dbProperties.getProperty(curPropKey));
        }

        // Hikari pool properties
        poolConfig.setMaximumPoolSize(m_poolMaxSize);
        poolConfig.setIdleTimeout(m_poolIdleTimeoutMillis);
        poolConfig.setConnectionTimeout(m_poolConnectTimeoutMillis);

        return new HikariPool(poolConfig);
    }

    // - Members ---------------------------------------------------------------

    private String m_driverClassName = null;

    private String m_jdbcUrl = null;

    private String m_username = null;

    private String m_password = null;

    private Properties m_dbProperties = null;

    private int m_poolMaxSize = DEFAULT_POOL_MAXSIZE;

    private long m_poolConnectTimeoutMillis = DEFAULT_POOL_CONNECTTIMEOUTMILLIS;

    private long m_poolIdleTimeoutMillis = DEFAULT_POOL_IDLETIMEOUTMILLIS;
}
