/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imagemetadata.impl;

import java.awt.Dimension;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.drew.imaging.ImageMetadataReader;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.MetadataException;
import com.openexchange.imageconverter.api.MetadataGroup;
import com.openexchange.imageconverter.api.MetadataKey;
import com.openexchange.imagetransformation.ImageMetadata;

/**
 * {@link MetadataReaderImpl}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class MetadataReaderImpl implements IMetadataReader {

    /**
     * Initializes a new {@link MetadataReaderImpl}.
     */
    public MetadataReaderImpl() {
        super();
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.api.ExifService#readExifData(java.io.InputStream)
     */
    @Override
    public IMetadata readMetadata(InputStream mediaInputStm)  throws MetadataException {
        MetadataImpl ret = null;

        if (null == mediaInputStm) {
            throw new MetadataException(new MetadataException("Argument must not be null"));
        }

        try {
            ret = new MetadataImpl(ImageMetadataReader.readMetadata(new BufferedInputStream(mediaInputStm), false));
        } catch (Throwable e) {
            throw new MetadataException(e);
        }

        return implUpdateMetadataHeader(ret, mediaInputStm);
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.api.ExifService#readExifData(java.io.File)
     */
    @Override
    public IMetadata readMetadata(File mediaInputFile) throws MetadataException {
        MetadataImpl metadata = null;

        if (null == mediaInputFile) {
            throw new MetadataException(new MetadataException("Argument must not be null"));
        }

        try {
            metadata = new MetadataImpl(ImageMetadataReader.readMetadata(mediaInputFile));
        } catch (@SuppressWarnings("unused") Throwable e) {
            // ok, file format may not support metadata => create default object
            metadata = new MetadataImpl();
        }

        try (final InputStream imageInputStm = FileUtils.openInputStream(mediaInputFile)) {
            return implUpdateMetadataHeader(metadata, imageInputStm);
        } catch (IOException e) {
            LOG.trace(Throwables.getRootCause(e).getMessage());
        }

        return null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.api.ExifService#readExifData(org.json.JSONObject)
     */
    @Override
    public IMetadata readMetadata(JSONObject jsonMetadata) throws MetadataException {
        return implUpdateMetadataHeader(new MetadataImpl(
            (null != jsonMetadata) ?
                jsonMetadata :
                    new JSONObject()), null);
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param metadata
     * @return
     */
    protected static MetadataImpl implUpdateMetadataHeader(@NonNull final MetadataImpl metadata, @Nullable final InputStream imageInputStm) {
        final MetadataGroupImpl headerGroup = (MetadataGroupImpl) metadata.getMetadataGroup(MetadataGroup.HEADER);

        if ((null == headerGroup) ||
            (headerGroup.getDataValue(MetadataKey.HEADER_PIXEL_WIDTH) == null) ||
            (headerGroup.getDataValue(MetadataKey.HEADER_PIXEL_HEIGHT) == null) ||
            (headerGroup.getDataValue(MetadataKey.HEADER_FORMAT_NAME) == null)) {

            Integer pixelWidth = null;
            Integer pixelHeight = null;
            String imageFormatName = null;

            if (null != imageInputStm) {
                try {
                    final ImageMetadata imageMetadata = MetadataUtils.getImageInformation(imageInputStm);
                    final Dimension dimension = imageMetadata.getDimension();

                    pixelWidth = Integer.valueOf(dimension.width);
                    pixelHeight = Integer.valueOf(dimension.height);
                    imageFormatName = imageMetadata.getFormatName();
                } catch (IOException e) {
                    LOG.trace(Throwables.getRootCause(e).getMessage());
                }
            }

            final Map<MetadataKey, Object> headerMap = new HashMap<>();

            headerMap.put(MetadataKey.HEADER_PIXEL_WIDTH, (null != pixelWidth) ? pixelWidth : Integer.valueOf(-1));
            headerMap.put(MetadataKey.HEADER_PIXEL_HEIGHT, (null != pixelHeight) ? pixelHeight : Integer.valueOf(-1));
            headerMap.put(MetadataKey.HEADER_FORMAT_NAME, (null != imageFormatName) ? imageFormatName : "png");

            metadata.getOrCreateMetadataGroup(MetadataGroup.HEADER).addMap(headerMap);
        }

        return metadata;
    }

    // Static members ----------------------------------------------------------

    final private static Logger LOG = LoggerFactory.getLogger(MetadataReaderImpl.class);
}
