/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.ElementLock;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItem;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageConverterPriority;
import com.openexchange.imageconverter.api.ImageFormat;
import com.openexchange.imageconverter.api.MetadataException;
import com.openexchange.imageconverter.api.MetadataImage;
import com.openexchange.imageconverter.impl.ImageProcessor.ImageProcessorStatus;

/**
 * {@link ImageConverterQueue}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
class ImageConverterQueue extends Thread {

    /**
     * {@link ImageConverterJobExecutor}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.2
     */
    private class ImageConverterJobExecutor {

        /**
         * Initializes a new {@link ImageConverterJobExecutor}.
         * @param maxThreadCount
         */
        ImageConverterJobExecutor(final int maxThreadCount) {
            super();

            m_jobExecutor = Executors.newFixedThreadPool(maxThreadCount);
            m_executorJobQueue = new LinkedBlockingQueue<>(maxThreadCount);

            for (int i = 0; i < maxThreadCount; ++i) {
                m_jobExecutor.execute(() -> {
                    while (isRunning()) {
                        try {
                            // Resource warning checked:
                            // job is closed after it has been run
                            final ImageConverterJob curJob = m_executorJobQueue.take();

                            if (null != curJob) {
                                final String imageKey = curJob.getImageKey();

                                ElementLocker.lock(imageKey, ElementLock.LockMode.STANDARD);

                                try {
                                    curJob.run();
                                } finally {
                                    // close job, imageKey will be removed from processingMap as well
                                    implCloseJob(curJob, !isRunning());

                                    ElementLocker.unlock(imageKey, ElementLock.UnlockMode.END_PROCESSING);
                                }
                            }
                        } catch (@SuppressWarnings("unused") InterruptedException e) {
                            // ok
                        }

                    }
                });
            }
        }

        /**
         *
         */
        public @Nullable List<Runnable> shutdown() {
            List<Runnable> ret = null;

            if (m_executorRunning.compareAndSet(true, false)) {
                ret = m_jobExecutor.shutdownNow();

                if (null != ret) {
                    ret.addAll(m_executorJobQueue);
                }
            }

            return ret;
        }

        /**
         * @param job
         */
        public boolean addJob(@NonNull final ImageConverterJob job) throws InterruptedException {
            if (m_executorRunning.get()) {
                m_executorJobQueue.put(job);
                return true;
            }

            return false;
        }

        /**
         * @param imageKey
         * @return true if one key has been removed
         */
        public boolean removeJob(@NonNull final String imageKey) {
            return implRemoveByPredicateOrAll((curImageKey) -> imageKey.equals(curImageKey));
        }

        /**
         * @return true if at least one key has been removed
         */
        public boolean removeAllOpenJobs() {
            return implRemoveByPredicateOrAll(null);
        }

        // - Implementation ----------------------------------------------------

        /**
         * @param predicateOrAll Remove all open jobs to execute, if given
         *  Predicate is null and only the selected ones if Predicate is given
         * @return true if at least one key has been removed
         */
        public boolean implRemoveByPredicateOrAll(@Nullable final Predicate<String> predicateOrAll) {
            boolean ret = false;

            synchronized (m_executorJobQueue) {
                for (final Iterator<ImageConverterJob> iter = m_executorJobQueue.iterator(); iter.hasNext();) {
                    final ImageConverterJob curJob = iter.next();

                    if ((null != curJob) && ((null == predicateOrAll) || predicateOrAll.test(curJob.getImageKey()))) {
                        ret = true;
                        iter.remove();

                        // close job, imageKey will be removed from processingMap as well
                        implCloseJob(curJob, true);
                    }
                }
            }

            return ret;
        }

        // - Members -----------------------------------------------------------

        final AtomicBoolean m_executorRunning = new AtomicBoolean(true);

        final private ExecutorService m_jobExecutor;

        final private LinkedBlockingQueue<ImageConverterJob> m_executorJobQueue;
    }

    /**
     * {@link QueueEntry}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.2
     */
    private class ImageConverterQueueEntry {

        /**
         * Initializes a new {@link ImageConverterQueueEntry}.
         * @param job
         * @param priority
         */
        ImageConverterQueueEntry(@NonNull final ImageConverterJob job, @NonNull final ImageConverterPriority priority) {
            m_job = job;
            m_priority = priority;
        }

        /**
         * @return
         */
        ImageConverterJob getJob() {
            return m_job;
        }

        /**
         * @return
         */
        ImageConverterPriority getPriority() {
            return m_priority;
        }

        // - Members -----------------------------------------------------------

        final private ImageConverterJob m_job;
        final private ImageConverterPriority m_priority;
    }

    @SuppressWarnings("serial")
    private class ImageConverterQueueImpl extends LinkedList<ImageConverterJob> {
        /**
         * Initializes a new {@link ImageConverterQueue}.
         */
        ImageConverterQueueImpl(ImageConverterPriority priority) {
            super();
            m_priority = priority;
        }

        /**
         * @return
         */
        ImageConverterPriority getPriority() {
            return m_priority;
        }

        // - Members -----------------------------------------------------------

        final private ImageConverterPriority m_priority;
    }

    /**
     * Initializes a new {@link ImageConverterQueue}.
     */
    ImageConverterQueue(@NonNull final ImageConverter imageConverter,
        @NonNull final IFileItemService fileItemService,
        @NonNull final IMetadataReader metadataReader) {

        super("ImageConverter Queue");

        m_imageConverter = imageConverter;
        m_fileItemService = fileItemService;
        m_metadataReader = metadataReader;

        // create an ordered BidiMap to map ImageFormats to image format strings and vice versa
        m_imageFormats = Arrays.asList(m_imageConverter.getImageFormats());
        m_imageProcessor = new ImageProcessor(
            ImageConverterConfig.IMAGECONVERTER_SEARCHPATH,
            ImageConverterConfig.IMAGECONVERTER_USE_GRAPHICSMAGICK,
            ImageConverterConfig.IMAGECONVERTER_IMAGE_USERCOMMENT);

        for (final ImageConverterPriority curPriority : ImageConverterPriority.values()) {
            m_queuesImpl.put(curPriority, new ImageConverterQueueImpl(curPriority));
        }

        m_executor = new ImageConverterJobExecutor(ImageConverterConfig.IMAGECONVERTER_THREAD_COUNT);

        if (LOG.isInfoEnabled()) {
            LOG.info(ImageConverterUtils.STR_BUILDER().
                append("IC conversion queue started (thread count: ").
                append(ImageConverterConfig.IMAGECONVERTER_THREAD_COUNT).
                append(", queue entries: ").append(ImageConverterConfig.IMAGECONVERTER_QUEUE_LENGTH).
                append(')').toString());
        }

    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Thread#run()
     */
    @Override
    public void run() {
        boolean trace = false;

        while (isRunning()) {
            trace = LOG.isTraceEnabled();

            try {
                // wait until a job is available in queue
                while (isRunning() && !implHasQueuedJob()) {
                    try {
                        m_lock.lock();
                        m_jobAvailableConditon.await();
                    } catch (@SuppressWarnings("unused") InterruptedException e) {
                        // ok
                    } finally {
                        m_lock.unlock();
                    }
                }

                // get next job from queue (prioritized)
                final ImageConverterQueueEntry queueEntry = implGetAndRemoveNextJob();

                if (null != queueEntry) {
                    // Resource warning has been checked:
                    // job is closed when runnble finishes or in case,
                    // the job could not be processed at all
                    final ImageConverterJob job = queueEntry.getJob();

                    ImageConverterUtils.IC_MONITOR.addQueueTimeMillis(
                        job.getCurrentMeasurementTimeMillis(),
                        queueEntry.getPriority());

                    if (!m_executor.addJob(job)) {
                        // close job, imageKey will be removed from processingMap as well
                        implCloseJob(job, true);
                    }
                }
            } catch (@SuppressWarnings("unused") Exception e) {
                if (trace) {
                    LOG.trace("IC queue received exception while taking next job from queue and forwarding to executor: {}", Throwables.getRootCause(e));
                }
            }
        }

        if (trace) {
            LOG.trace("IC queue worker finished");
        }
    }

    // - Public API ------------------------------------------------------------

    /**
     * @return
     */
    public ImageProcessorStatus getImageProcessorStatus() {
        return m_imageProcessor.getStatus();
    }

    /**
     * @return
     */
    public ImageConverter getImageConverter() {
        return m_imageConverter;
    }

    /**
     * @return
     */
    public IFileItemService getFileItemService() {
        return m_fileItemService;
    }

    /**
     * @return
     */
    public IMetadataReader getMetadataReader() {
        return m_metadataReader;
    }

    /**
     * @return
     */
    public ImageProcessor getImageProcessor() {
        return m_imageProcessor;
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     */
    void cache(@NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @Nullable final String context,
        @Nullable ImageConverterPriority priority) throws ImageConverterException {

        if (!m_imageConverter.isAvailable(imageKey)) {
            implCache(imageKey, srcImageStm, null, context, null);
        }
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     * @return
     */
    MetadataImage cacheAndGetMetadataImage(@NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @NonNull final String requestFormat,
        @Nullable final String context,
        @Nullable final ImageConverterPriority priority) throws ImageConverterException {

        MetadataImage ret = null;

        try {
            ret = getMetadataImage(imageKey, requestFormat);
        } catch (ImageConverterException e) {
            LOG.trace("IC not yet able to read MetadataImage, waiting for processed key first", e);
        }

        if (null == ret) {
            return implCache(
                imageKey,
                srcImageStm,
                requestFormat,
                context,
                (null != priority) ? priority : ImageConverterPriority.MEDIUM);
        }

        return ret;
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     * @return
     */
    MetadataImage getMetadataImage(@NonNull final String imageKey, @NonNull final String requestFormat) throws ImageConverterException {
        if (m_imageConverter.isAvailable(imageKey)) {
            ImageFormat requestTargetFormat = null;
            final boolean trace = LOG.isTraceEnabled();

            try {
                final IMetadata imageMetadata = getMetadata(imageKey);

                if (null == imageMetadata) {
                    throw new MetadataException("IC could not read metadata");
                }

                // lock element in processing mode (just lock, no condition wait)
                // first and check, if appropriate job is still queued or processed
                try {
                    ElementLocker.lock(imageKey, ElementLock.LockMode.STANDARD);

                    final ImageConverterQueueEntry queueEntry = implGetQueuedJob(imageKey, false);

                    // job is queued =>
                    // promote job to highest priority queue, process requested format
                    // and return result immediately;
                    // job is asynchronously processed afterwards
                    if (null != queueEntry) {
                        if (trace) {
                            LOG.trace("IC queue synchronously processes job in user request: {}", imageKey);
                        }

                        // change priority for already scheduled job
                        implChangeJobPriority(imageKey, ImageConverterPriority.highest());

                        // get MetadataImage from job directly
                        return queueEntry.getJob().processRequestFormat(requestFormat);
                    }
                } finally {
                    ElementLocker.unlock(imageKey, ElementLock.UnlockMode.STANDARD);
                }

                try {
                    // image key is available in general, but not queued anymore =>
                    // wait until outstanding job has been processed completely, if necessary
                    if (trace) {
                        LOG.trace("IC queue gets image/waits for finish of asynchronous job processing in user request: {}", imageKey);
                    }

                    ElementLocker.lock(imageKey, ElementLock.LockMode.WAIT_IF_PROCESSED);

                    // get all processed image formats and determine target request format
                    final List<ImageFormat> availableImageFormats = implGetAvailableImageKeyFormats(imageKey);

                    // if there're no available formats, the entry must be invalid
                    if (0 != availableImageFormats.size()) {
                        // determine target format to retrieve
                        requestTargetFormat = ImageConverterUtils.getBestMatchingFormat(
                            availableImageFormats,
                            imageMetadata,
                            ImageFormat.parseImageFormat(requestFormat));

                        if (null != requestTargetFormat) {
                            return new MetadataImage(ImageConverterUtils.readImageByTargetFormat(m_fileItemService, imageKey, requestTargetFormat), imageMetadata);
                        }
                    }

                    throw new ImageConverterException("IC could not determine any valid MetadataImage based on given request format");
                } finally {
                    ElementLocker.unlock(imageKey, ElementLock.UnlockMode.STANDARD);
                }
            } catch (Exception e) {
                if (trace) {
                    LOG.trace(ImageConverterUtils.STR_BUILDER().
                        append("IC queue could not get MetadataImage for ").
                        append(imageKey).append(": ").
                        append(requestFormat).append(" / ").
                        append((null != requestTargetFormat) ? requestTargetFormat.getFormatString() : "n/a").toString(),
                        Throwables.getRootCause(e));
                }
            }
        }

        return null;
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     * @return
     */
    IMetadata getMetadata(@NonNull final String imageKey) throws ImageConverterException {
        if (m_imageConverter.isAvailable(imageKey)) {
            // job is queued =>
            // get metadadata from still running job
            final ImageConverterJob job = m_processingMap.get(imageKey);

            // Resource warning has been checked:
            // job is closed when runnable finishes;
            // try to get getmetada from still scheduled or processing job
            if (null != job) {
                return job.getImageMetadata();
            }

            // image key is available in general, but not in queue anymore =>
            // read from persistent data
            ElementLocker.lock(imageKey, ElementLock.LockMode.WAIT_IF_PROCESSED);

            try {
                return ImageConverterUtils.readMetadata(m_fileItemService, m_metadataReader, imageKey);
            } finally {
                ElementLocker.unlock(imageKey, ElementLock.UnlockMode.STANDARD);
            }
        }

        return null;
    }

    /**
     * @param imageKey
     * @throws ImageConverterException
     */
    void removeImageConverterJob(@NonNull final String imageKey) {
        ElementLocker.lock(imageKey, ElementLock.LockMode.STANDARD);

        // lock element in processing mode (just lock, no condition wait)
        // first and check, if appropriate job is still queued or processed
        try {
            final boolean trace = LOG.isTraceEnabled();
            String removeLocation = null;

            if (null != implGetQueuedJob(imageKey, true)) {
                if (trace) {
                    removeLocation = "priority queue";
                }
            } else {
                // remove job from executor queue
                // removal of persistent data should be handled by
                // caller providing a list of imageKeys to remove
                // for performance reasons
                m_executor.removeJob(imageKey);

                if (trace) {
                    removeLocation = "executor queue";
                }
            }

            if (trace) {
                LOG.trace(ImageConverterUtils.STR_BUILDER().
                    append("IC queue removed pending job from ").
                    append(removeLocation).append(": ").
                    append(imageKey).toString());
            }
        } finally {
            ElementLocker.unlock(imageKey, ElementLock.UnlockMode.END_PROCESSING);
        }

   }

    /**
     *
     */
    void shutdown() {
        if (m_running.compareAndSet(true, false)) {
            final long shutdownStartTimeMillis = System.currentTimeMillis();
            final boolean trace = LOG.isTraceEnabled();

            if (trace) {
                LOG.trace("IC queue starting shutdown");
            }

            try {
                final List<Runnable> openJobList = m_executor.shutdown();

                if (null != openJobList) {
                    if (trace) {
                        LOG.trace("IC is closing already queued entries in shutdown: {}", openJobList.size());
                    }

                    // cleanup/close of all pending Runnables
                    openJobList.forEach((curJob) -> {
                        // close job, imageKey will be removed from processingMap as well
                        implCloseJob(((ImageConverterJob) curJob), true);
                    });
                }

                interrupt();
                m_imageProcessor.shutdown();
            } catch (@SuppressWarnings("unused") Exception e) {
                Thread.currentThread().interrupt();
            }

            LOG.trace("IC queue shutdown open Runnable count after shutdown: {}{}", m_processingMap.size());
            LOG.trace("IC queue shutdown finished in {}ms", Long.valueOf((System.currentTimeMillis() - shutdownStartTimeMillis)));
        }
    }

    /**
     * @return
     */
    boolean isRunning() {
        return m_running.get() && !interrupted();
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param imageKey
     */
    protected MetadataImage implCache(@NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @Nullable final String requestFormat,
        @Nullable final String context,
        @Nullable final ImageConverterPriority priority) throws ImageConverterException {

        // we need to ensure that no other job with the same
        // imageKey is currently running => wait until processed
        ElementLocker.lock(imageKey, ElementLock.LockMode.WAIT_IF_PROCESSED);

        try {
            // begin processing out of the lock state of the previous lock
            ElementLocker.lock(imageKey, ElementLock.LockMode.BEGIN_PROCESSING);
        } finally {
            // unlock the first waiting lock now, the BEGIN_PROCESSING lock
            // is still active and will be unlocked temporarily after
            // queuing and initial request processing;
            // final END_PROCESSING unlock will happen in an async. way
            ElementLocker.unlock(imageKey, ElementLock.UnlockMode.STANDARD);
        }

        if (m_processingMap.containsKey(imageKey)) {
            LOG.warn("IC queue processing map already contains imageKey while start queueing: {}", imageKey);
        }

        ImageConverterJob job = null;
        MetadataImage ret = null;
        boolean errorOccured = false;

        try {
            job = new ImageConverterJob(imageKey, srcImageStm, m_imageFormats, this, context);

            if (null != requestFormat) {
                errorOccured = (null == (ret = job.processRequestFormat(requestFormat)));
            }

            // add job to appropriate queue for asynchronous
            // processing, if everything went well so far
            if (!errorOccured) {
                implAdd(job, (null != priority) ? priority : ImageConverterPriority.BACKGROUND);
            }
        } catch (ImageConverterException e) {
            // job is closed in finally block
            errorOccured = true;
            throw e;
        } finally {
            if (errorOccured) {
                // close job in case of error =>
                // the imageKey will be removed from processingMap as well
                if (null != job) {
                    implCloseJob(job, true);
                }

                LOG.warn("IC removed entry after occurence of an error: {}", imageKey);
            }

            // unlock image key with unlock mode based on error status
            ElementLocker.unlock(imageKey,
                errorOccured ?
                    ElementLock.UnlockMode.END_PROCESSING :
                        ElementLock.UnlockMode.STANDARD);
        }

        // Resource warning has been checked:
        // job is already closed or will be closed
        // when ImageConverter Job finishes
        return ret;
    }

    /**
     * @param imageKey
     * @return
     * @throws ImageConverterException
     */
    List<ImageFormat> implGetAvailableImageKeyFormats(@NonNull final String imageKey) throws ImageConverterException {
        IFileItem[] fileItems = null;

        try {
            fileItems = m_fileItemService.get(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey);
        } catch (FileItemException e) {
            throw new ImageConverterException(e);
        }

        if (null != fileItems) {
            final List<ImageFormat> imageFormats = new ArrayList<>(fileItems.length);

            for (final IFileItem curFileItem : fileItems) {
                final String fileId = curFileItem.getFileId();

                if ((null != fileId) && !fileId.startsWith(ImageConverterUtils.IMAGECONVERTER_METADATA_FILEID)) {
                    imageFormats.add(ImageFormat.parseImageFormat(fileId));
                }
            }

            return imageFormats;
        }

        return EMPTY_IMAGEFORMAT_LIST;
    }

    /**
     * @return
     * @throws ImageConverterException
     */
    void implCloseJob(@NonNull final ImageConverterJob job, final boolean removePersistentData) {
        final String imageKey = job.getImageKey();

        try {
            if (removePersistentData) {
                m_fileItemService.removeSubGroup(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey);
            }
        } catch (FileItemException e) {
            LOG.trace("IC error while removing imageKey when closing Runnable: {}", imageKey, e);
        } finally {
            // close job itself
            IOUtils.closeQuietly(job);

            // remove imageKey from processing map
            if (null != imageKey) {
                m_processingMap.remove(imageKey);
            }
        }

        LOG.trace("IC queue closed job: {}", imageKey);
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param priority
     * @param job
     */
    void implAdd(@NonNull ImageConverterJob job, @NonNull final ImageConverterPriority priority) {
        int updatedQueueSize = 0;

        // add job to appropriate, prioritized
        // reset measurement time of job in order to retrieve the
        // queue time, after which the job is taken from queue
        job.initMeasurementTime();

        synchronized (m_queuesImpl) {
            final ImageConverterQueueImpl queue = m_queuesImpl.get(priority);

            queue.add(job);
            updatedQueueSize = queue.size();
        }

        ImageConverterUtils.IC_MONITOR.setQueueCount(priority, updatedQueueSize);

        if (LOG.isTraceEnabled()) {
            LOG.trace(ImageConverterUtils.STR_BUILDER().
                append("IC entry count after put to ").
                append(priority.name().toUpperCase()).
                append(" queue: ").append(updatedQueueSize).toString());
        }

        m_processingMap.put(job.getImageKey(), job);
        implSignalAvailability();
    }

    /**
     * @param job
     * @param newPriority
     * @return
     */
    private boolean implChangeJobPriority(@NonNull final String imageKey, @NonNull final ImageConverterPriority newPriority) {
        boolean priorityChange = false;

        synchronized (m_queuesImpl) {
            // iterate over all prioritized queues from highest to lowest
            // priority, in order to retrieve the next job candidate
            for (final ImageConverterPriority curPriority : PRIORITY_LOW_TO_HIGH) {
                // don't remove/add from/to queue with same priority
                if (curPriority != newPriority) {
                    final ImageConverterQueueImpl curQueue = m_queuesImpl.get(curPriority);
                    final Iterator<ImageConverterJob> iter = curQueue.iterator();

                    while (iter.hasNext()) {
                        // Resource warning has been checked:
                        // job is closed when runnble finishes
                        final ImageConverterJob curJob = iter.next();

                        if (imageKey.equals(curJob.getImageKey())) {
                            // move from old priority queue into new priority queue
                            iter.remove();
                            m_queuesImpl.get(newPriority).add(curJob);

                            if (LOG.isTraceEnabled()) {
                                LOG.trace(ImageConverterUtils.STR_BUILDER().
                                        append("IC entry has been moved from ").
                                        append(curPriority.name().toUpperCase()).
                                        append(" queue to ").
                                        append(newPriority.name().toUpperCase()).
                                        append(" queue: ").
                                        append(imageKey).toString());
                            }

                            priorityChange = true;
                            break;
                        }
                    }
                }
            }
        }

        if (priorityChange) {
            implSignalAvailability();
        }

        return priorityChange;
    }

    /**
     * @return
     */
    private boolean implHasQueuedJob() {
        synchronized (m_queuesImpl) {
            // iterate over all prioritzed queues to check if there's a waiting job
            for (final ImageConverterPriority curPriority : PRIORITY_ORDER_HIGH_TO_LOW) {
                if (!m_queuesImpl.get(curPriority).isEmpty()) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * @param imageKey
     * @return
     */
    private ImageConverterQueueEntry implGetQueuedJob(@NonNull final String imageKey, boolean endProcessing) {
        synchronized (m_queuesImpl) {
            ImageConverterJob curJob = null;

            // iterate over all prioritzed queues to check if there's a waiting job
            for (final ImageConverterPriority curPriority : PRIORITY_ORDER_HIGH_TO_LOW) {
                final ImageConverterQueueImpl implQueue = m_queuesImpl.get(curPriority);

                for (final Iterator<ImageConverterJob> iter = implQueue.iterator(); iter.hasNext();) {
                    // resource warning has been checked:
                    // job is closed later or it will be closed
                    // after it will have been executed as FutureTask
                    if ((null != (curJob = iter.next())) && imageKey.equals(curJob.getImageKey())) {
                        if (endProcessing) {
                            iter.remove();

                            // close job, imageKey will be removed from processingMap as well
                            implCloseJob(curJob, false);
                        }

                        // resource warning has been checked, job was either
                        // closed already or it will be closed after it will
                        // have been executed as FutureTask
                        return new ImageConverterQueueEntry(curJob, curPriority);
                    }
                }
            }
        }

        return null;
    }

    /**
     * @return
     */
    protected ImageConverterQueueEntry implGetAndRemoveNextJob() {
        synchronized (m_queuesImpl) {
            ImageConverterJob job = null;

            // iterate over all prioritzed queues from highest to lowest
            // priority, in order to retrieve the next job candidate
            for (final ImageConverterPriority curPriority : PRIORITY_ORDER_HIGH_TO_LOW) {
                final ImageConverterQueueImpl curQueue = m_queuesImpl.get(curPriority);

                if (!curQueue.isEmpty() && (null != (job = curQueue.remove()))) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace(ImageConverterUtils.STR_BUILDER().
                                append("IC entry count after take from ").
                                append(curPriority.name().toUpperCase()).
                                append(" queue: ").append(curQueue.size()).toString());
                    }

                    // resource warning has been checked, job is closed
                    // after it will have been executed as FutureTask
                    return new ImageConverterQueueEntry(job, curPriority);
                }
            }
        }

        return null;
    }

    /**
     * Signal thread waiting on this condition to continue processing
     *
     */
    void implSignalAvailability() {
        try {
            m_lock.lock();
            m_jobAvailableConditon.signal();
        } finally {
            m_lock.unlock();
        }
    }

    // - Static members --------------------------------------------------------

    /**
     * EMPTY_IMAGEFORMAT_LIST
     */
    final private static List<ImageFormat> EMPTY_IMAGEFORMAT_LIST = new ArrayList<>();

    /**
     * PRIORITY_ORDER_HIGH_TO_LOW
     */
    final private static ImageConverterPriority[] PRIORITY_ORDER_HIGH_TO_LOW = {
        ImageConverterPriority.INSTANT,
        ImageConverterPriority.MEDIUM,
        ImageConverterPriority.BACKGROUND
    };

    /**
     * PRIORITY_LOW_TO_HIGH
     * clone PRIORITY_ORDER_HIGH_TO_LOW and reverse in static init.
     */
    final private static ImageConverterPriority[] PRIORITY_LOW_TO_HIGH = (ImageConverterPriority[]) ArrayUtils.clone(PRIORITY_ORDER_HIGH_TO_LOW);

    static {
        ArrayUtils.reverse(PRIORITY_LOW_TO_HIGH);
    }

    // - Members ---------------------------------------------------------------

    final protected ConcurrentMap<String, ImageConverterJob> m_processingMap = new ConcurrentHashMap<>();

    final private ImageConverter m_imageConverter;

    final private ReentrantLock m_lock = new ReentrantLock(true);

    final private Condition m_jobAvailableConditon = m_lock.newCondition();

    final private IFileItemService m_fileItemService;

    final private IMetadataReader m_metadataReader;

    final private List<ImageFormat> m_imageFormats;

    final private Map<ImageConverterPriority, ImageConverterQueueImpl> m_queuesImpl = new HashMap<>();

    final private ImageConverterJobExecutor m_executor;

    final private AtomicBoolean m_running = new AtomicBoolean(true);

    final private ImageProcessor m_imageProcessor;
}
