package com.openexchange.fileitem.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;

/**
 * {@link ConnectionWrapper}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.3
 */
public class ConnectionWrapper implements AutoCloseable {

    /**
     * {@link ConnectionClosedHandler}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.3
     */
    public interface ConnectionClosedHandler {
        void closed(Connection con);
    }

    /**
     * Initializes a new {@link ConnectionWrapper}.
     * @param connection
     */
    ConnectionWrapper(final Connection connection) {
        m_connection = connection;
    }

    /**
     * Initializes a new {@link ConnectionWrapper}.
     * @param connection
     */
    ConnectionWrapper(final Connection connection, final ConnectionClosedHandler connectionClosedHandler) {
        m_connection = connection;
        m_connectionClosedHandler = connectionClosedHandler;
    }

    /**
     *
     */
    @Override
    public void close() throws Exception {
        if (null != m_connection) {
            m_connection.close();

            synchronized (this) {
                if (null != m_connectionClosedHandler) {
                    m_connectionClosedHandler.closed(m_connection);
                }
            }
        }
    }

    // Public API --------------------------------------------------------------

    /**
     * setConnectionClosedHandler
     *
     * @param connectionClosedHandler
     */
    void setConnectionClosedHandler(final ConnectionClosedHandler connectionClosedHandler) {
        synchronized (this) {
            m_connectionClosedHandler = connectionClosedHandler;
        }
    }

    /**
     * isValid
     *
     * @return
     */
    public boolean isValid() {
        return (null != m_connection);
    }

    /**
     * getConnection
     *
     * @return
     */
    public Connection getConnection() {
        return m_connection;
    }

    /**
     * prepareStatement
     *
     * @param sql
     * @return
     * @throws SQLException
     */
    public PreparedStatement prepareStatement(final String sql) throws SQLException {
        return (null != m_connection) ? m_connection.prepareStatement(sql) : null;
    }

    /**
     * createStatement
     *
     * @return
     * @throws SQLException
     */
    public Statement createStatement() throws SQLException {
        return (null != m_connection) ? m_connection.createStatement() : null;
    }

    /**
     * commit
     *
     * @throws SQLException
     */
    public void commit() throws SQLException {
        if (null != m_connection) {
            m_connection.commit();
        }
    }

    // - Members ---------------------------------------------------------------

    private final Connection m_connection;

    private ConnectionClosedHandler m_connectionClosedHandler = null;
}
