/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.clt;

import java.net.URI;
import java.net.URISyntaxException;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.glassfish.jersey.client.ClientConfig;
import org.json.JSONObject;
import com.openexchange.cli.AbstractRestCLI;

/**
 * {@link ImageConverterAdmin}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.1
 */
public class ImageConverterAdmin extends AbstractRestCLI<Void> {

    protected static final String DELETE_LONG  = "delete";
    protected static final String DELETE_SHORT = "d";

    protected static final String METRICS_LONG  = "metrics";
    protected static final String METRICS_SHORT = "m";

    protected static final String CONTEXT_LONG    = "context";
    protected static final String CONTEXT_SHORT   = "c";
    protected static final String CONTEXT_DEFAULT = "";

    protected static final String ENDPOINT_LONG    = "api-root";
    protected static final String ENDPOINT_DEFAULT = "http://localhost:8005/imageconverter/";

    /**
     * @param args
     */
    public static void main(String[] args) {
        new ImageConverterAdmin().execute(args);
    }

    /**
     * Initializes a new {@link ImageConverterAdmin}.
     */
    public ImageConverterAdmin() {
        super();
    }

    // - AbstractRestCLI -----------------------------------------------------

    @Override
    protected Boolean requiresAdministrativePermission() {
        return Boolean.TRUE;
    }

    /**
     *
     */
    @Override
    protected void checkArguments(CommandLine cmd) {
        // nothing to do
    }

    /**
    *
    */
    @Override
    protected void checkOptions(CommandLine cmd) {
        // nothing to do
    }

    /**
    *
    */
    @Override
    protected String getFooter() {
        return "ImageConverter administration.";
    }

    /**
    *
    */
    @Override
    protected String getName() {
        return "imageconverter-admin -U myUser:myPassword [OPTIONS]";
    }

    /**
     *
     */
    @Override
    protected String getHeader() {
        return "imageconverter-admin -U myUser:myPassword [-m | -d [-c context]] [--api-root URL]";
    }

    /**
     *
     */
    @Override
    protected void addOptions(Options options) {
        options.addOption(METRICS_SHORT, METRICS_LONG, false, "Show the current metrics of the running server.");
        options.addOption(DELETE_SHORT, DELETE_LONG, false, "Delete all image keys on the running server, add optional \"-c context\" to restrict removal to that context");

        options.addOption(CONTEXT_SHORT, CONTEXT_LONG, true, "The optional context of the keys that should be removed. Default: n/a.");

        options.addOption(null, ENDPOINT_LONG, true, " URL to an alternative HTTP API endpoint. Example: 'https://192.168.0.1:8005/imageconverter/'");
    }

    /**
     *
     */
    @Override
    protected WebTarget getEndpoint(CommandLine cmd) {
        final String endpoint = cmd.getOptionValue(ENDPOINT_LONG, ENDPOINT_DEFAULT);
        WebTarget ret = null;

        try {
            final URI uri = new URI(endpoint);
            final ClientConfig config = new ClientConfig();

            ret = ClientBuilder.newClient(config).target(uri);

            if (cmd.hasOption(METRICS_SHORT)) {
                ret = ret.path("status").queryParam("metrics", "true");
            } else if (cmd.hasOption(DELETE_SHORT)) {
                final String context = cmd.getOptionValue(CONTEXT_SHORT);

                ret = ret.path("clearImages");

                if ((null != context) && (context.length() > 0)) {
                    ret = ret.queryParam("context", context);
                }
            }
        } catch (URISyntaxException e) {
            System.err.print("Unable to return endpoint: " + e.getMessage());
        }

        return ret;
    }

    /**
     *
     */
    @Override
    protected Void invoke(Options option, CommandLine cmd, Builder context) throws Exception {
        context.accept(MediaType.APPLICATION_JSON_TYPE, MediaType.APPLICATION_OCTET_STREAM_TYPE, MediaType.TEXT_PLAIN_TYPE);

        final String response = cmd.hasOption(DELETE_SHORT) ? context.delete(String.class) : context.get(String.class);
        final JSONObject entity = new JSONObject(response);

        System.out.println("Operation result: " + entity.toString());

        return null;
    }
}
