/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import java.awt.Dimension;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemWriteAccess;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageFormat;
import com.openexchange.imageconverter.api.MetadataException;
import com.openexchange.imageconverter.api.MetadataImage;
import com.openexchange.imagetransformation.ScaleType;

/**
 * {@link ImageConverterJob}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.2
 */
class ImageConverterJob implements Runnable, Closeable {

    /**
     * Initializes a new {@link ImageConverterJob}.
     * @param imageKey
     * @param srcImageStm
     * @param imageFormats
     * @param queue
     * @param context
     * @param metadata
     * @throws ImageConverterException
     */
    ImageConverterJob(
        @NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @NonNull final List<ImageFormat> imageFormats,
        @NonNull ImageConverterQueue queue,
        @Nullable final String context,
        @Nullable final IMetadata metadata ) throws ImageConverterException {

        super();

        m_queue = queue;
        m_imageKey = imageKey;
        m_context = context;

        try {
            // create tmp. input file from input stream, file is removed by calling close method
            m_inputFile = File.createTempFile("oxic", ".tmp", ImageConverterConfig.IMAGECONVERTER_SPOOLPATH);
            FileUtils.copyInputStreamToFile(srcImageStm, m_inputFile);

            if ((null == m_inputFile) || (m_inputFile.length() < 1)) {
                throw new IOException("IC ConversionJob creation of job not possible due to missing or empty input file: " + imageKey);
            }

            // assign given metadata or read metadata from input file
            m_imageMetadata = (null != metadata) ?
                metadata :
                    implCreateKeyMetadata(queue.getMetadataReader(), m_imageKey, m_inputFile, m_context);

            if (null == m_imageMetadata) {
                throw new MetadataException("IC ConversionJob error while trying to read image metadata from input file: " + imageKey);
            }
        } catch (IOException | MetadataException e) {
            close();
            throw new ImageConverterException("IC ConversionJob error while creating ImageConverterJob", e);
        }

        m_imageFormats = implCreateImageFormatList(m_imageMetadata, imageFormats);
        implFillProcessMap();
    }

    /*
     * (non-Javadoc)
     *
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() {
        FileUtils.deleteQuietly(m_inputFile);
        m_inputFile = null;
        m_formatsToProcess.clear();
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        final boolean trace = LOG.isTraceEnabled();
        ImageFormat curTargetFormat = null;

        if (trace) {
            LOG.trace("IC ConversionJob run started: " + m_imageKey);
        }

        try {
            final List<ImageFormat> workingImageFormatsList = new LinkedList<>();

            // use a working list to allow arbitrary removal of entries from m_formatsToProcess
            // map while iterating over all open working List entries, created from map's keySet
            workingImageFormatsList.addAll(m_formatsToProcess.keySet());

            for (final Iterator<ImageFormat> iter = workingImageFormatsList.iterator(); iter.hasNext();) {
                curTargetFormat = iter.next();

                if (trace) {
                    LOG.trace(ImageConverterUtils.STR_BUILDER().
                        append("IC ConversionJob worker thread asynchronously processing ").
                        append(m_imageKey).append(": ").
                        append(curTargetFormat.getFormatString()).toString());
                }

                implProcessKey(curTargetFormat, true);
            }
        } catch (Exception e) {
            LOG.error(ImageConverterUtils.STR_BUILDER().
                append(m_imageKey).append(": ").
                append((null != curTargetFormat) ? curTargetFormat.getFormatString() :"null").toString(), e);
        } finally {
            ImageConverterUtils.IC_MONITOR.incrementProcessedKeys(m_processingTime);

            if (trace) {
                LOG.trace("IC ConversionJob run finished: " + m_imageKey);
            }
        }
    }

    /**
     * @return
     */
    protected String getImageKey() {
        return m_imageKey;
    }

    /**
     *
     */
    protected void initMeasurementTime() {
        m_measurementStartTimeMillis = System.currentTimeMillis();
    }

    /**
     * @return
     */
    protected long getCurrentMeasurementTimeMillis() {
        return (System.currentTimeMillis() - m_measurementStartTimeMillis);
    }

    /**
     *
     */
    protected IMetadata getImageMetadata() {
        return m_imageMetadata;
    }

    /**
     * Process requested
     *
     * @param targetFormat
     * @return
     * @throws ImageConverterException
     */
    protected void implProcessKey(@NonNull final ImageFormat targetFormat, final boolean isAsync) throws ImageConverterException {
        ImageFormat processingFormat = m_formatsToProcess.get(targetFormat);
        final long subProcessingStartTimeMillis = System.currentTimeMillis();

        if (null == processingFormat) {
            // check first, if we need to create the reference format entry for the requested ScaleType
            final ImageFormat referenceFormat = m_referenceFormatMap.get(targetFormat.getScaleType());

            if ((null != referenceFormat) && (null != (processingFormat = m_formatsToProcess.get(referenceFormat)))) {
                implCreateThumbnail(m_imageKey, m_inputFile, m_imageMetadata, referenceFormat, processingFormat, m_context, isAsync);
                m_formatsToProcess.remove(referenceFormat);
            }

            implCreatReferenceThumbnail(m_imageKey, m_imageMetadata, targetFormat, referenceFormat, m_context, isAsync);
        } else {
            implCreateThumbnail(m_imageKey, m_inputFile, m_imageMetadata, targetFormat, processingFormat, m_context, isAsync);
        }

        m_formatsToProcess.remove(targetFormat);
        m_processingTime += (System.currentTimeMillis() - subProcessingStartTimeMillis);
    }

    /**
     * @param targetFormat
     * @return
     * @throws ImageConverterException
     */
    protected MetadataImage processRequestFormat(@NonNull final String requestFormat) throws ImageConverterException {
        ImageFormat requestTargetFormat = null;
        MetadataImage ret = null;

        try {
            requestTargetFormat = ImageConverterUtils.getBestMatchingFormat(
                m_imageFormats,
                m_imageMetadata,
                ImageFormat.parseImageFormat(requestFormat));

            if (null == requestTargetFormat) {
                throw new ImageConverterException("IC ConversionJob is not able to parse requested image format: " + requestFormat);
            }

            // no need to do anything, if requested format has already been processed
            if (m_formatsToProcess.containsKey(requestTargetFormat)) {
                implProcessKey(requestTargetFormat, false);
            }

            ret = new MetadataImage(ImageConverterUtils.readImageByTargetFormat(m_queue.getFileItemService(), m_imageKey, requestTargetFormat), m_imageMetadata);
        } catch (Exception e) {
            throw new ImageConverterException(ImageConverterUtils.STR_BUILDER().
                append(m_imageKey).
                append(" / ").append(requestFormat).
                append(" / ").append((null != requestTargetFormat) ?
                    requestTargetFormat.getFormatString() :
                        "null").toString(), e);
        }

        return ret;
    }

    // - Implementation --------------------------------------------------------

    /**
    *
    */
    private void implFillProcessMap() {
        // The process map is filled with 3 different kinds of key/value pairs:
        // 1. key and value are (!=null) and same: do a real image processing with the key format and store under key format
        // 2. key and value are (!=null) but different: do real image processing with the value format, but store under key format
        // 3. key is (!=null) and value is (==null): set only the reference property and store under key format without real image processing

        final Dimension imageDimension = m_imageMetadata.getImageDimension();

        for (final ImageFormat curImageFormat : m_imageFormats) {
            final ScaleType curScaleType = curImageFormat.getScaleType();
            final ImageFormat referenceFormat = m_referenceFormatMap.get(curScaleType);

            if (null != referenceFormat) {
                // if a reference format is set, we need to check, if
                // the current format image result matches the
                // reference format result, in order to properly
                // use the reference format
                if (ImageConverterUtils.shouldUseReferenceFormat(imageDimension, curImageFormat)) {
                    m_formatsToProcess.put(curImageFormat, null);
                } else {
                    m_formatsToProcess.put(curImageFormat, curImageFormat);
                }

            } else if (ImageConverterUtils.shouldUseReferenceFormat(imageDimension, curImageFormat)) {
                // the imageFormat at the first reference format position is substituted with the original size format
                final ImageFormat originalFormat = ImageFormat.createFrom(curImageFormat.getFormatShortName(),
                    true,
                    imageDimension.width,
                    imageDimension.height,
                    curImageFormat.getScaleType(),
                    false,
                    curImageFormat.getQuality());

                m_referenceFormatMap.put(curScaleType, curImageFormat);
                m_formatsToProcess.put(curImageFormat, originalFormat);
            } else {
                m_formatsToProcess.put(curImageFormat, curImageFormat);
            }
        }
    }

    /**
     * @return
     * @throws MetadataException
     */
    protected IMetadata implCreateKeyMetadata(@NonNull final IMetadataReader metadataReader, @NonNull final String imageKey, @NonNull final File inputFile, @Nullable final String context) throws MetadataException {
        IMetadata ret = metadataReader.readMetadata(inputFile);

        if (null != ret) {
            try (final IFileItemWriteAccess fileWriteAcess = m_queue.getFileItemService().getWriteAccess(
                ImageConverterUtils.IMAGECONVERTER_GROUPID,
                imageKey,
                ImageConverterUtils.IMAGECONVERTER_METADATA_FILEID)) {

                if (null != fileWriteAcess) {
                    try (final PrintWriter printWriter = new PrintWriter(fileWriteAcess.getOutputStream())) {
                        printWriter.write(ret.getJSONObject().toString());
                    }

                    implSetProperties(fileWriteAcess, context, null);

                    if (LOG.isTraceEnabled()) {
                        final Dimension imageDimension = ret.getImageDimension();
                        final String formatName = ret.getImageFormatName();

                        LOG.trace(ImageConverterUtils.STR_BUILDER().
                            append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                            append(formatName).append(" image: ").
                            append(imageKey).toString());
                    }
                }
            } catch (Exception e) {
                LOG.error("IC error while creating metadata entry: {}", imageKey, e);

                implRemoveFile(imageKey, ImageConverterUtils.IMAGECONVERTER_METADATA_FILEID);
                ret = null;
            }
        }

        return ret;
    }

    /**
     * @param writeAccess
     * @param referencePropertyValue
     * @throws FileItemException
     */
    protected static void implSetProperties(@NonNull final IFileItemWriteAccess writeAccess, @Nullable final String context, @Nullable final String referencePropertyValue) throws FileItemException {
        if (null != context) {
            writeAccess.setKeyValue(ImageConverterUtils.IMAGECONVERTER_KEY_CONTEXT, context);
        }

        if (null != referencePropertyValue) {
            writeAccess.setKeyValue(ImageConverterUtils.IMAGECONVERTER_KEY_REFERENCE, referencePropertyValue);
        }
    }

    /**
     * @param targetFormat
     * @param processingFormat
     * @throws ImageConverterException
     */
    protected void implCreateThumbnail(@NonNull final String imageKey, @NonNull final File inputFile,
        @NonNull final IMetadata imageMetadata,
        @NonNull final ImageFormat targetFormat,
        @NonNull final ImageFormat processingFormat,
        @Nullable final String context,
        final boolean isAsync) throws ImageConverterException {

        final String targetFormatStr = targetFormat.getFormatString();
        final ImageFormat usedProcessingFormat = (null != processingFormat) ? processingFormat : targetFormat;
        boolean succeeded = false;

        try (final IFileItemWriteAccess fileWriteAccess =
            m_queue.getFileItemService().getWriteAccess(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey, targetFormatStr)) {

            if ((null != fileWriteAccess) && m_queue.getImageProcessor().scale(inputFile, fileWriteAccess.getOutputFile(), usedProcessingFormat)) {
                implSetProperties(fileWriteAccess, context, null);
                succeeded = true;
            }

            if (LOG.isTraceEnabled()) {
                final Dimension imageDimension = imageMetadata.getImageDimension();
                final String imageFormatShortName = imageMetadata.getImageFormatName();

                LOG.trace(ImageConverterUtils.STR_BUILDER().
                    append("IC ").append(isAsync ? "a" : "").
                    append("synchronously created ").append(imageFormatShortName).append(" thumbnail for ").
                    append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                    append(imageFormatShortName).append(" image: ").
                    append(imageKey).append(" / ").append(targetFormatStr).
                    append(" (Used physical format: ").append(usedProcessingFormat.getFormatString()).append(')').toString());
            }
        } catch (Exception e) {
            final Dimension imageDimension = imageMetadata.getImageDimension();
            final String imageFormatShortName = imageMetadata.getImageFormatName();

            LOG.error(ImageConverterUtils.STR_BUILDER().
                append("IC error while ").append(isAsync ? "a" : "").
                append("synchronously creating ").append(imageFormatShortName).append(" thumbnail for ").
                append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] image: ").
                append(imageKey).append(" / ").append(targetFormatStr).
                append(" (Used physical format: ").append(usedProcessingFormat.getFormatString()).append(')').toString(), e);
        } finally {
            if (!succeeded) {
                implRemoveFile(imageKey, targetFormatStr);
            }
        }
    }

    /**
     * @param targetFormat
     * @throws ImageConverterException
     */
    protected void implCreatReferenceThumbnail(@NonNull final String imageKey,
        @NonNull final IMetadata imageMetadata,
        @NonNull final ImageFormat targetFormat,
        @NonNull final ImageFormat referenceFormat,
        @Nullable final String context,
        final boolean isAsync) throws ImageConverterException {

        // we create a reference entry in case the shrinkOnly Flag
        // is set and the scaling would expand the source format
        final String targetFormatStr = targetFormat.getFormatString();
        final String referenceFormatStr = referenceFormat.getFormatString();

        try (final IFileItemWriteAccess fileWriteAccess = m_queue.
            getFileItemService().
            getWriteAccess(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey, targetFormatStr)) {

            implSetProperties(fileWriteAccess, context, referenceFormatStr);

            if (LOG.isTraceEnabled()) {
                final Dimension imageDimension = imageMetadata.getImageDimension();
                final String imageFormatShortName = imageMetadata.getImageFormatName();

                LOG.trace(ImageConverterUtils.STR_BUILDER().
                    append("IC ").append(isAsync ? "a" : "").
                    append("synchronously created reference entry for ").
                    append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                    append(imageFormatShortName).append(" image: ").
                    append(imageKey).append(" / ").append(targetFormatStr).
                    append(" (Used reference format: ").append(referenceFormatStr).append(')').toString());
            }
        } catch (Exception e) {
            implRemoveFile(imageKey, targetFormatStr);

            final Dimension imageDimension = imageMetadata.getImageDimension();
            final String imageFormatShortName = imageMetadata.getImageFormatName();

            LOG.error(ImageConverterUtils.STR_BUILDER().
                append("IC error while ").append(isAsync ? "a" : "").
                append("synchronously creating reference entry for: ").
                append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                append(imageFormatShortName).append(" image: ").
                append(imageKey).append(" / ").append(targetFormatStr).
                append(" (Used reference format: ").append(referenceFormatStr).append(')').toString(), e);
        }
    }

    /**
     * @param imageKey
     * @param fileId
     */
    protected void implRemoveFile(@NonNull final String imageKey, @NonNull final String fileName) {
        try {
            m_queue.getFileItemService().remove(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey, fileName);
        } catch (FileItemException e) {
            if (LOG.isTraceEnabled()) {
                LOG.trace(ImageConverterUtils.STR_BUILDER().
                    append(imageKey).append(": ").
                    append(fileName).toString(), e);
            }
        }
    }

    /**
     * @param metadata
     * @param imageFormats
     * @return
     */
    protected static List<ImageFormat> implCreateImageFormatList(@NonNull final IMetadata metadata, @NonNull final List<ImageFormat> imageFormats) {
        final List<ImageFormat> ret = new ArrayList<>();
        final String imageFormatShortName = metadata.getImageFormatName();
        final boolean isAnimated = imageFormatShortName.startsWith("gif");
        final boolean isTransparent = isAnimated || imageFormatShortName.startsWith("png") || imageFormatShortName.startsWith("tif");

        for (final ImageFormat curFormat : imageFormats) {
            ImageFormat.ImageType targetFormatImageType = null;


            switch (curFormat.getFormatShortName()) {
                case "jpg": {
                    targetFormatImageType = ImageFormat.ImageType.JPG;
                    break;
                }

                case "png": {
                    targetFormatImageType = ImageFormat.ImageType.PNG;
                    break;
                }

                case "auto":
                default: {
                    targetFormatImageType = isAnimated ?
                        // TODO (KA) add GIF format to possible target types
                        ImageFormat.ImageType.PNG :
                            (isTransparent  ? ImageFormat.ImageType.PNG : ImageFormat.ImageType.JPG);
                    break;
                }
            }

            final ImageFormat targetImageFormat = new ImageFormat(targetFormatImageType);

            targetImageFormat.setAutoRotate(curFormat.isAutoRotate());
            targetImageFormat.setWidth(curFormat.getWidth());
            targetImageFormat.setHeight(curFormat.getHeight());
            targetImageFormat.setScaleType(curFormat.getScaleType());
            targetImageFormat.setShrinkOnly(curFormat.isShrinkOnly());


            ret.add(targetImageFormat);
        }

        return ret;
    }

    // - Members ---------------------------------------------------------------

    final private String m_imageKey;

    final private List<ImageFormat> m_imageFormats;

    final private ImageConverterQueue m_queue;

    final private String m_context;

    final private IMetadata m_imageMetadata;

    final private Map<ImageFormat, ImageFormat> m_formatsToProcess = Collections.synchronizedMap(new LinkedHashMap<>());

    final private Map<ScaleType, ImageFormat> m_referenceFormatMap = new HashMap<>(ScaleType.values().length);

    private File m_inputFile;

    private long m_measurementStartTimeMillis = 0;

    private long m_processingTime = 0;
}
