/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.server.rest;

import static org.apache.commons.lang.ArrayUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.security.PermitAll;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.ResponseBuilder;
import javax.ws.rs.core.Response.Status;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.NumberUtils;
import org.apache.commons.lang.StringUtils;
import org.glassfish.jersey.media.multipart.BodyPart;
import org.glassfish.jersey.media.multipart.BodyPartEntity;
import org.glassfish.jersey.media.multipart.MultiPart;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.IImageConverter;
import com.openexchange.imageconverter.api.IImageConverterMonitoring;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageServerUtil;
import com.openexchange.imageconverter.api.MetadataImage;
import com.openexchange.imageconverter.impl.ImageConverterUtils;
import com.openexchange.rest.services.JAXRSService;
import com.openexchange.server.ServiceLookup;


/**
 * {@link ImageConverterServerRest}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
@Path("/imageconverter/")
@Produces("application/json")
@PermitAll
public class ImageConverterServerRest extends JAXRSService {

    /**
     * OX_IMAGECONVERTER_SERVER_REMOTE_API_VERSION
     */
    public static String OX_IMAGECONVERTER_SERVER_REMOTE_API_VERSION = "1";

    /**
     * Initializes a new {@link ImageConverterServerRest}.
     * @param serviceLookup
     */
    public ImageConverterServerRest(@NonNull final ServiceLookup serviceLookup, @NonNull final IImageConverter imageConverter) {
        super(serviceLookup);

        m_imageConverter = imageConverter;
    }

    /**
     * @param context
     * @return
     * @throws IOException
     */
    @GET
    @Path("/status/")
    @Produces(MediaType.TEXT_HTML)
    public String status(@QueryParam("metrics") @DefaultValue("false") String metrics) {
        LOG.trace("IC WebService received #status request");

        return implCreateImageConverterServerHtmlPage(m_running.get() ?
            OX_IMAGECONVERTER_SERVER_DEFAULT_RESPONSE_TEXT :
                OX_IMAGECONVERTER_SERVER_TERMINATED_RESPONSE_TEXT,
            StringUtils.equalsIgnoreCase(metrics, "true") ?
                ImageConverterUtils.IC_MONITORING :
                    null);
    }

    // - IImageConverter -------------------------------------------------------

    /**
     * @param imageKey
     * @param requestFormat
     * @param context
     * @return
     */
    @GET
    @Path("/getImage/{imageKey}/{requestFormat}")
    @Produces(MediaType.MULTIPART_FORM_DATA)
    public Response getImage(@PathParam("imageKey") String imageKey,
        @PathParam("requestFormat") String requestFormat,
        @QueryParam("context") @DefaultValue("") String context) {

        LOG.trace("IC WebService received #getImage request: {} / {}", imageKey, requestFormat);

        ResponseBuilder responseBuilder = null;

        if (m_running.get()) {
            try (final InputStream inputStm = m_imageConverter.getImage(imageKey, requestFormat, context)) {
                responseBuilder = implCreateFileResponse(inputStm);
            } catch (Exception e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.trace("IC WebService error in #getImage", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }

    /**
     * @param imageKey
     * @param requestFormat
     * @param context
     * @return
     */
    @GET
    @Path("/getMetadata/{imageKey}")
    @Produces(MediaType.MULTIPART_FORM_DATA)
    public Response getMetadata(@PathParam("imageKey") String imageKey,
        @QueryParam("context") @DefaultValue("") String context) {

    	LOG.trace("IC WebService received #getMetadata request: {}", imageKey);

        ResponseBuilder responseBuilder = null;

        if (m_running.get()) {
            try {
                final IMetadata metadata = m_imageConverter.getMetadata(imageKey, context);

                try (final MetadataImage metadataImage = new MetadataImage(null, metadata)) {
                    responseBuilder = implCreateMetadataImageResponse(metadataImage);
                }
            } catch (Exception e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.error("IC WebService error in #getMetadata", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }

    /**
     * @param imageKey
     * @param requestFormat
     * @param context
     * @return
     */
    @GET
    @Path("/getImageAndMetadata/{imageKey}/{requestFormat}")
    @Produces(MediaType.MULTIPART_FORM_DATA)
    public Response getImageAndMetadata(@PathParam("imageKey") String imageKey,
        @PathParam("requestFormat") String requestFormat,
        @QueryParam("context") @DefaultValue("") String context) {

    	LOG.trace("IC WebService received #getImageAndMetadata request: {} / {}", imageKey, requestFormat);

        ResponseBuilder responseBuilder = null;

        if (m_running.get()) {
            try (final MetadataImage metadataImage = m_imageConverter.getImageAndMetadata(imageKey, requestFormat, context)) {
                responseBuilder = implCreateMetadataImageResponse(metadataImage);
            } catch (Exception e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.error("IC WebService error in #getImageAndMetadata", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }

    /**
     * @param imageKey
     * @param context
     * @param srcImageStm
     */
    @PUT
    @Path("/cacheImage/{imageKey}")
    @Consumes(MediaType.MULTIPART_FORM_DATA)
    public Response cacheImage(MultiPart multiPart,
        @PathParam("imageKey") String imageKey,
        @QueryParam("context") @DefaultValue("") String context) {

    	LOG.trace("IC WebService received #cacheImage request: {}", imageKey);

        ResponseBuilder responseBuilder = null;

        if (m_running.get()) {
            try (final InputStream inputStm = implGetInputStream(multiPart)) {
                if (null != inputStm) {
                    m_imageConverter.cacheImage(imageKey, inputStm, context);
                    responseBuilder = implCreateSuccessResponse();
                } else {
                    throw new IllegalArgumentException("IC #cacheImage server input file is not available");
                }
            } catch (Exception e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.error("IC WebService error in #cacheImage", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }

    /**
     * @param imageKey
     * @param requestFormat
     * @param context
     * @param srcImageStm
     * @return
     */
    @POST
    @Path("cacheAndGetImage/{imageKey}/{requestFormat}")
    @Consumes(MediaType.MULTIPART_FORM_DATA)
    @Produces(MediaType.MULTIPART_FORM_DATA)
    public Response cacheAndGetImage(MultiPart multiPart,
        @PathParam("imageKey") String imageKey,
        @PathParam("requestFormat") String requestFormat,
        @QueryParam("context") @DefaultValue("") String context) {

        LOG.trace("IC WebService received #cacheAndGetImage request: {} / {}", imageKey, requestFormat);

        ResponseBuilder responseBuilder = null;

        if (m_running.get()) {
            try (final InputStream inputStm = implGetInputStream(multiPart)) {
                if (null != inputStm) {
                    try (final InputStream resultInputStm = m_imageConverter.cacheAndGetImage(imageKey, requestFormat, inputStm, context)) {
                        responseBuilder = implCreateFileResponse(resultInputStm);
                    }
                } else {
                    throw new IllegalArgumentException("IC #cacheAndGetImage server input file is not available");
                }
            } catch (Exception e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.error("IC WebService error in #cacheAndGetImage", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }

    /**
     * @param imageKey
     * @param requestFormat
     * @param context
     * @param srcImageStm
     * @return
     */
    @POST
    @Path("cacheAndGetImageAndMetadata/{imageKey}/{requestFormat}")
    @Consumes(MediaType.MULTIPART_FORM_DATA)
    @Produces(MediaType.MULTIPART_FORM_DATA)
    public Response cacheAndGetImageAndMetadata(MultiPart multiPart,
        @PathParam("imageKey") String imageKey,
        @PathParam("requestFormat") String requestFormat,
        @QueryParam("context") @DefaultValue("") String context) {

        LOG.trace("IC WebService received #cacheAndGetImageAndMetadata request: {} / {}", imageKey, requestFormat);

        ResponseBuilder responseBuilder = null;

        if (m_running.get()) {
            try (final InputStream sourceImageInputStream = implGetInputStream(multiPart)) {
                if (null != sourceImageInputStream) {
                    try (final MetadataImage metadaImage = m_imageConverter.cacheAndGetImageAndMetadata(imageKey, requestFormat, sourceImageInputStream, context)) {
                        responseBuilder = implCreateMetadataImageResponse(metadaImage);
                    }
                } else {
                    throw new IllegalArgumentException("IC WebService input file is not available");
                }
            } catch (Exception e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.error("IC WebService error in #cacheAndGetImageAndMetadata", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }

    /**
     * @param contextValue
     */
    @DELETE
    @Path("/clearImages")
    public Response clearImages(@QueryParam("context") @DefaultValue("") String context) {
        ResponseBuilder responseBuilder = null;

        LOG.trace("IC WebService received #clearImages request");

        if (m_running.get()) {
            try {
                if (isEmpty(context)) {
                    m_imageConverter.clearImages();
                } else {
                    m_imageConverter.clearImages(context);
                }

                responseBuilder = implCreateSuccessResponse();
            } catch (ImageConverterException e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.error("IC WebService error in #clearImages", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }


    /**
     * @param imageKeyValue
     */
    @DELETE
    @Path("/clearImagesByKey/{imageKey}")
    public Response clearImagesByKey(@PathParam("imageKey") String imageKey) {
        ResponseBuilder responseBuilder = null;

        LOG.trace("IC WebService received #clearImagesByKey request: {}", imageKey);

        if (m_running.get()) {
            try {
                m_imageConverter.clearImagesByKey(imageKey);
                responseBuilder = implCreateSuccessResponse();
            } catch (ImageConverterException e) {
                responseBuilder = implCreateExceptionResponse(e);
                LOG.error("IC WebService error in #clearImagesByKey", e);
            }
        } else {
            responseBuilder = implCreateImageConverterErrorResponse("ImageConverter server terminated!");
        }

        return responseBuilder.build();
    }

    /**
     * @param contextValue
     * @return
     */
    @GET
    @Path("/getKeyCount")
    @Produces(MediaType.APPLICATION_JSON)
    public JSONObject getKeyCount(@QueryParam("context") @DefaultValue("") String context) {
        JSONObject ret = new JSONObject();

        LOG.trace("IC WebService received #getKeyCount request");

        if (m_running.get()) {
            try {
                ret.put("count", m_imageConverter.getKeyCount(context));
            } catch (Exception e) {
            	LOG.error("IC WebService error in #getKeyCount", e);
            }
        } else {
            try {
                ret = new JSONObject(implCreateJSONResponse(CODE_ERROR_TERMINATED, MESSAGE_ERROR_TERMINATED));
            } catch (Exception e) {
                LOG.error("IC WebService error in #getKeyCount", e);
            }
        }

        return ret;
    }

    /**
     * @param context
     * @return
     * @throws ImageConverterException
     */
    @GET
    @Path("/getKeys")
    @Produces(MediaType.APPLICATION_JSON)
    public JSONObject getKeys(@QueryParam("context") @DefaultValue("") String context) {
        JSONObject ret = new JSONObject();

        if (m_running.get()) {
            LOG.trace("IC WebService received #getKey request");

            try {
                ret.put("keys", Arrays.asList(m_imageConverter.getKeys(context)));
            } catch (Exception e) {
            	LOG.error("IC WebService error in #getKeys", e);
            }
        } else {
            try {
                ret = new JSONObject(implCreateJSONResponse(CODE_ERROR_TERMINATED, MESSAGE_ERROR_TERMINATED));
            } catch (Exception e) {
            	LOG.error("IC WebService error in #getKeys", e);
            }
        }

        return ret;
    }

    /**
     *
     */
    public void shutdown() {
        if (m_running.compareAndSet(true, false)) {
            LOG.trace("ImageConverter server shutdown finished");
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param content
     * @return
     */
    private static String implCreateImageConverterServerHtmlPage(final String[] content, @Nullable final IImageConverterMonitoring monitoring) {
        final StringBuilder htmlBuilder = new StringBuilder(256);

        htmlBuilder.append("<html>").
            append("<head><meta charset=\"UTF-8\"><title>").append(OX_IMAGECONVERTER_SERVER_TITLE).append("</title></head>").
            append("<body><h1 align=\"center\">").append(OX_IMAGECONVERTER_SERVER_TITLE).append("</h1>");

        // print server Id in every case
        htmlBuilder.append("<p>Id: ").append(ImageServerUtil.IMAGE_SERVER_ID).append("</p>");

        // print API version in every case
        htmlBuilder.append("<p>API: v").append(OX_IMAGECONVERTER_SERVER_REMOTE_API_VERSION).append("</p>");

        if (isNotEmpty(content)) {
            for (int i = 0; i < content.length;) {
                htmlBuilder.append("<p>").append(content[i++]).append("</p>");
            }
        }

        if (null != monitoring) {
            htmlBuilder.append("<br />").append("<p><u>Metrics</u></p>");

            for (Method method : monitoring.getClass().getMethods())
                if (null != method) {
                    final String methodName = method.getName();

                    if (isNotEmpty(methodName) && StringUtils.startsWithIgnoreCase(methodName, "get") && (method.getParameterTypes().length == 0)) {
                        final String metricsPropertyName = methodName.substring(3);
                        final boolean isRatioMetric = metricsPropertyName.endsWith("Ratio");

                        try {
                            final String numberString = new StringBuilder(256).append(method.invoke(monitoring)).toString();

                            if (NumberUtils.isNumber(numberString)) {
                                htmlBuilder.append("<p>").append(metricsPropertyName).append(": ");

                                if (isRatioMetric) {
                                    htmlBuilder.append(Double.parseDouble(numberString));
                                } else {
                                    htmlBuilder.append(Long.parseLong(numberString));
                                }

                                htmlBuilder.append("</p>");
                            }
                        } catch (@SuppressWarnings("unused") Exception e) {
                            //
                        }
                    }
                }
        }

        return htmlBuilder.append("</body></html>").toString();
    }

    /**
     * @param inputStm
     * @return
     */
    private static ResponseBuilder implCreateFileResponse(final InputStream inputStm) {
        return (null != inputStm) ? Response.ok(inputStm) : Response.status(Status.NOT_FOUND);
    }

    /**
     * @param inputStm
     * @return
     */
    private static ResponseBuilder implCreateMetadataImageResponse(final MetadataImage metadataImage) {
        ResponseBuilder ret = null;

        if (null != metadataImage) {
            final MultiPart multipart = new MultiPart(MediaType.MULTIPART_FORM_DATA_TYPE);

            try (final InputStream imageInputStm = metadataImage.getImageInputStream()) {
                if (null != imageInputStm) {
                    {
                        final BodyPart imageBodyPart = new BodyPart(MediaType.APPLICATION_OCTET_STREAM_TYPE);

                        imageBodyPart.setEntity(imageInputStm);
                        multipart.bodyPart(imageBodyPart);
                    }
                }
            } catch (IOException e) {
                ret = implCreateExceptionResponse(e);
            }

            if (null == ret) {
                final IMetadata imageMetadata = metadataImage.getMetadata();

                if (null != imageMetadata) {
                    final BodyPart metadataBodyPart = new BodyPart(MediaType.APPLICATION_JSON_TYPE);

                    metadataBodyPart.setEntity(imageMetadata.getJSONObject());
                    multipart.bodyPart(metadataBodyPart);
                }

                ret = Response.ok(multipart);
            }
        } else {
            ret = implCreateImageConverterErrorResponse(null);
        }

        return ret;
    }

    /**
     * @return
     */
    private static ResponseBuilder implCreateSuccessResponse() {
        return Response.
            ok().
            type(MediaType.APPLICATION_JSON).
            entity(implCreateJSONResponse(CODE_SUCCESS, MESSAGE_SUCCESS));
    }

    /**
     * @return
     */
    private static ResponseBuilder implCreateImageConverterErrorResponse(String message) {
        return Response.
            serverError().
            type(MediaType.APPLICATION_JSON).
            entity(implCreateJSONResponse(CODE_ERROR_IMAGECONVERTER, isNotEmpty(message) ? message : MESSAGE_ERROR_IMAGECONVERTER_GENERAL));
    }

    /**
     * @return
     */
    private static ResponseBuilder implCreateTerminateResponse(String message) {
        return Response.
            serverError().
            type(MediaType.APPLICATION_JSON).
            entity(implCreateJSONResponse(CODE_ERROR_TERMINATED, isNotEmpty(message) ? message : MESSAGE_ERROR_TERMINATED));
    }

    /**
     * @param e
     * @return
     */
    private static ResponseBuilder implCreateExceptionResponse(final Exception e) {
        return Response.
            serverError().
            type(MediaType.APPLICATION_JSON).
            entity(implCreateJSONResponse(CODE_ERROR_IMAGECONVERTER_SERVER, Throwables.getRootCause(e).getMessage()));
    }

    /**
     * @param code
     * @param message
     * @return
     */
    private static String implCreateJSONResponse(@NonNull final Integer code, final String message) {
        final JSONObject responseObject = new JSONObject();

        try {
            responseObject.put("code", code);
            responseObject.put("message", isNotEmpty(message) ? message : MESSAGE_SUCCESS);
        } catch (JSONException e) {
        	LOG.error(e.getMessage(), e);
        }

        return responseObject.toString();
    }

    /**
     * @param multiPart
     * @return
     */
    private static InputStream implGetInputStream(@NonNull final MultiPart multiPart) {
        InputStream ret = null;

        try {
            for (final BodyPart part : multiPart.getBodyParts()) {
                final Object entity = part.getEntity();

                if (entity instanceof BodyPartEntity) {
                    ret = ((BodyPartEntity) entity).getInputStream();
                } else if (entity instanceof File) {
                    ret = new BufferedInputStream(FileUtils.openInputStream((File) entity));
                }

                if (null != ret) {
                    break;
                }
            }
        } catch (IOException e) {
            LOG.error(e.getMessage(), e);
        }

        return ret;
    }

    // - Static Members --------------------------------------------------------

    final private static Logger LOG = LoggerFactory.getLogger(ImageConverterServerRest.class);

    final private static Integer CODE_SUCCESS = Integer.valueOf(0);

    final private static String MESSAGE_SUCCESS = StringUtils.EMPTY;

    final private static Integer CODE_ERROR_IMAGECONVERTER_SERVER = Integer.valueOf(1);

    final private static Integer CODE_ERROR_IMAGECONVERTER = Integer.valueOf(2);

    final private static Integer CODE_ERROR_TERMINATED = Integer.valueOf(99);

    final private static String MESSAGE_ERROR_TERMINATED = "ImageConverter terminated!";

    final private static String MESSAGE_ERROR_IMAGECONVERTER_GENERAL = "ImageConverter general error";

    final private static String OX_IMAGECONVERTER_SERVER_TITLE = "OX Software GmbH Image Converter Server";

    final private static String[] OX_IMAGECONVERTER_SERVER_DEFAULT_RESPONSE_TEXT = { "Error Code: 0", " Status: running..." };

    final private static String[] OX_IMAGECONVERTER_SERVER_TERMINATED_RESPONSE_TEXT = { "Error Code: 99", " terminated!" };

    // - Members ---------------------------------------------------------------

    final private  IImageConverter m_imageConverter;

    final private AtomicBoolean m_running = new AtomicBoolean(true);
}
