/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.impl;

import java.io.File;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.io.FileUtils;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.FileItemException;

/**
 * {@link FileStoreRemover}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileStoreRemover extends Thread {

    final private static String CLEANUP_FILENAME = "filestoredata.bin";

    final private static int MAX_REMOVE_FILE_COUNT = 4096;

    /**
     * Initializes a new {@link FileStoreRemover}.
     */
    public FileStoreRemover(@NonNull FileItemService fileItemService) {
        super("FileItemService FileStoreRemover");

        m_fileItemService = fileItemService;
        m_cleanupFile = new File(FileItemUtils.getDataDir(), CLEANUP_FILENAME);
    }

    /* (non-Javadoc)
     * @see java.lang.Thread#run()
     */
    @Override
    public void run() {
        final ArrayList<FileStoreData> curRemoveList = new ArrayList<>(MAX_REMOVE_FILE_COUNT);

        // read all left entries from last shutdown first
        // and fill removal list with these entries
        implReadLeftOverFileStoreData();

        while (isRunning()) {
            final boolean trace = FileItemUtils.isLogTrace();

            synchronized (m_fileStoreDataList) {
                if (!m_fileStoreDataList.isEmpty()) {
                    curRemoveList.ensureCapacity(MAX_REMOVE_FILE_COUNT);

                    for (final ListIterator<FileStoreData> iter = m_fileStoreDataList.listIterator(); iter.hasNext() && (curRemoveList.size() < MAX_REMOVE_FILE_COUNT);) {
                        // move entry from toDeleteList to work list
                        curRemoveList.add(iter.next());
                        iter.remove();
                    }
                }
            }

            // perform removal of FileStore files in work list
            if (!curRemoveList.isEmpty()) {
                try {
                    final int removedCount = FileItemUtils.deleteFileStoreFiles(m_fileItemService, curRemoveList);

                    if (trace) {
                        FileItemUtils.logTrace(new StringBuilder(128).
                            append("FileStoreRemover removed ").
                            append(removedCount).append(" files (").
                            append(m_fileStoreDataList.size()).
                            append(" files left)").toString());
                    }
                } catch (FileItemException e) {
                    FileItemUtils.logExcp(e);
                } finally {
                    curRemoveList.clear();
                }
            }

            synchronized (m_fileStoreDataList) {
                if (m_fileStoreDataList.isEmpty()) {
                    try {
                        m_fileStoreDataList.wait(3000);
                    } catch (@SuppressWarnings("unused") InterruptedException e) {
                        // ok
                    }
                }
            }
        }

        // make all left over entries to be deleted persistent after shutdown,
        // in order to be processed the next time the service is started
        implWriteLeftOverFileStoreData();
    }

    // - Public API ------------------------------------------------------------

    /**
     *
     */
    public void shutdown() {
        if (m_running.compareAndSet(true, false)) {
            interrupt();
        }
    }

    /**
     * @return
     */
    public boolean isRunning() {
        return m_running.get() && !interrupted();
    }

    /**
     * @param fileStoreData
     */
    public int addFileStoreDataToRemove(@NonNull final FileStoreData fileStoreData) {
        synchronized (m_fileStoreDataList) {
            m_fileStoreDataList.add(fileStoreData);
            m_fileStoreDataList.notify();
        }

        return 1;
    }

    /**
     * @param fileStoreData
     */
    public int addFileStoreDatasToRemove(@NonNull final FileStoreData[] fileStoreDatas) {
        synchronized (m_fileStoreDataList) {
            final int sizeBefore = m_fileStoreDataList.size();

            for (FileStoreData curFileStoreData : fileStoreDatas) {
                if (null != curFileStoreData) {
                    m_fileStoreDataList.add(curFileStoreData);
                }
            }

            m_fileStoreDataList.notify();

            return (m_fileStoreDataList.size() - sizeBefore);
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     *
     */
    void implReadLeftOverFileStoreData() {
        if (m_cleanupFile.canRead()) {
            try (final ObjectInputStream objInputStm = new ObjectInputStream(FileUtils.openInputStream(m_cleanupFile))) {
                final Object readObject = objInputStm.readObject();
                int size = 0;

                if (readObject instanceof Collection<?>) {
                    synchronized (m_fileStoreDataList) {
                        final int sizeBefore = m_fileStoreDataList.size();

                        m_fileStoreDataList.addAll((Collection<FileStoreData>) readObject);
                        size = m_fileStoreDataList.size() - sizeBefore;
                    }

                    if (FileItemUtils.isLogTrace()) {
                        FileItemUtils.logTrace(new StringBuilder(128).
                            append("FileStoreRemover read ").
                            append(size).
                            append(" files to be removed").toString());
                    }
                } else {
                    FileItemUtils.logError("Unexpected type of object when reading cleanupFile, ecxpected List<FileStoreData>");
                }
            } catch (Exception e) {
                FileItemUtils.logError("Cleanup file could not be read: " + Throwables.getRootCause(e));
            }
        }
    }

    /**
     *
     */
    void implWriteLeftOverFileStoreData() {
        if (m_cleanupFile.getParentFile().canWrite()) {
            try (final ObjectOutputStream objOutputStm = new ObjectOutputStream(FileUtils.openOutputStream(m_cleanupFile))) {
                int size = 0;

                synchronized (m_fileStoreDataList) {
                    size = m_fileStoreDataList.size();
                    objOutputStm.writeObject(m_fileStoreDataList);
                }

                if (FileItemUtils.isLogTrace()) {
                    FileItemUtils.logTrace(new StringBuilder(128).
                        append("FileStoreRemover wrote ").
                        append(size).
                        append(" files to be removed after restart").toString());
                }
            } catch (Exception e) {
                FileItemUtils.logError("Cleanup file could not be written: " + Throwables.getRootCause(e));
            }
        } else {
            FileItemUtils.logError("Cleanup file could not be written due to missing write permissions to: " + m_cleanupFile.getAbsolutePath());
        }
    }

    // - Members ---------------------------------------------------------------

    final private AtomicBoolean m_running = new AtomicBoolean(true);

    final private FileItemService m_fileItemService;

    final private File m_cleanupFile;

    private LinkedList<FileStoreData> m_fileStoreDataList = new LinkedList<>();
}
