/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import static org.apache.commons.lang.ArrayUtils.isEmpty;
import static org.apache.commons.lang.ArrayUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.lang.ArrayUtils;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.ElementLock;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItem;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.IImageConverter;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageFormat;
import com.openexchange.imageconverter.api.MetadataImage;
import com.openexchange.imageconverter.impl.ImageConverterMonitoring.RequestType;
import com.openexchange.osgi.annotation.SingletonService;

/**
 * {@link ImageConverter}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
/**
 * {@link ImageConverter}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public final class ImageConverter implements IImageConverter {

    /**
     * Initializes a new {@link ImageConverter}.
     *
     * @param fileItemService
     * @param config
     */
    @SingletonService
    public ImageConverter(@NonNull final IFileItemService fileItemService,
        @NonNull final IMetadataReader metadataReader,
        @NonNull final ImageConverterConfig imageConverterConfig) throws ImageConverterException {

        super();

        m_fileItemService = fileItemService;
        m_metadataReader = metadataReader;
        m_imageFormats = imageConverterConfig.getImageFormats();

        // the given image Format list has to contains at least one valid entry
        if (m_imageFormats.length < 1) {
            throw new ImageConverterException("IC detected ImageFormat array with no valid entries => ensure at least one valid ImageFormat contained");
        }

        Arrays.sort(m_imageFormats);

        try {
            // register custom keys at FileItemService
            fileItemService.registerGroup(ImageConverterUtils.IMAGECONVERTER_GROUPID, ImageConverterUtils.IMAGECONVERTER_KEYS);
        } catch (FileItemException e) {
            throw new ImageConverterException("IC error when registering customs keys at IFileItemService", e);
        }

        // create and start worker queue to perform the image transformations (either synchronously or asynchrnously)
        (m_imageConversionQueue = new ImageConverterQueue(this, m_fileItemService, metadataReader)).start();

        // finally trace list of used image formats
        if (LOG.isInfoEnabled()) {
            final StringBuilder infoStr = new StringBuilder("IC ImageConverter is using the following target formats:\n");

            for (final ImageFormat curImageFormat : m_imageFormats) {
                infoStr.append("\tIC ").append(curImageFormat.toString()).append('\n');
            }

            LOG.info(infoStr.toString());
        }

        // initialize ImageConverterUtils with this singleton object
        ImageConverterUtils.init(this);

        (m_keyManager = new ImageConverterCache(this)).start();
    }

    /**
     *
     */
    public void shutdown() {
        if (m_terminated.compareAndSet(false, true)) {
            final long shutdownStartTimeMillis = System.currentTimeMillis();

            LOG.info("IC starting shutdown");

            m_keyManager.shutdown();
            m_imageConversionQueue.shutdown();

            LOG.info("IC shutdown finished in {}ms", (System.currentTimeMillis() - shutdownStartTimeMillis));
        }
    }

    /**
     * @return
     */
    public boolean isTerminated() {
        return m_terminated.get();
    }

    // - IImageConverter ----------------------------------------------------------

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#getImage(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public InputStream getImage(final String imageKey, final String requestFormat, final String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace(new StringBuilder("IC #getImage called (imageKey / requestFormat): ").
            append(imageKey).append(" / ").append(requestFormat).toString(),
            context);

        ImageConverterStatus status = (isNotEmpty(imageKey) && isNotEmpty(requestFormat)) ?
            ImageConverterStatus.OK :
                ImageConverterStatus.KEY_NOT_VALID;


        try (final MetadataImage metadataImage = m_imageConversionQueue.getMetadataImage(imageKey, requestFormat, implGetContext(context))) {
            InputStream ret = (null != metadataImage) ?
                metadataImage.getImageInputStream(true) :
                    null;

            if (null == ret) {
                status = ImageConverterStatus.KEY_NOT_AVAILABLE;
            }

            return ret;
        } catch (IOException | ImageConverterException e) {
            LOG.trace("IC exception while getting MetadataImage", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw new ImageConverterException("IC error while getting MetadataImage: " + imageKey, e);
        } finally {
            implFinishRequest(RequestType.GET, requestStartTime, status);
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getMetadata(java.lang.String, java.lang.String[])
     */
    @Override
    public IMetadata getMetadata(String imageKey, String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace(new StringBuilder("IC #getMetadata called (imageKey / requestFormat): ").
            append(imageKey).toString(),
            context);

        ImageConverterStatus status = isNotEmpty(imageKey) ?
            ImageConverterStatus.OK :
                ImageConverterStatus.KEY_NOT_VALID;

        try {
            final IMetadata ret = (ImageConverterStatus.OK == status) ?
                m_imageConversionQueue.getMetadata(imageKey, implGetContext(context)) :
                    null;

            if (null == ret) {
                status = ImageConverterStatus.KEY_NOT_AVAILABLE;
            }

            return ret;
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while getting Metadata", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.GET, requestStartTime, status);
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getImageAndMetadata(java.lang.String, java.lang.String, java.lang.String[])
     */
    @Override
    public MetadataImage getImageAndMetadata(String imageKey, String requestFormat, String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace(new StringBuilder("IC #getImageAndMetadata called (imageKey / requestFormat): ").
            append(imageKey).append(" / ").append(requestFormat).toString(),
            context);

        ImageConverterStatus status = isNotEmpty(imageKey) ?
            ImageConverterStatus.OK :
                ImageConverterStatus.KEY_NOT_VALID;

        try {
            final MetadataImage ret = (ImageConverterStatus.OK == status) ?
                m_imageConversionQueue.getMetadataImage(imageKey, requestFormat, implGetContext(context)) :
                    null;

            if (null == ret) {
                status = ImageConverterStatus.KEY_NOT_AVAILABLE;
            }

            return ret;
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while getting ImageAndMetadata", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.GET, requestStartTime, status);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#cacheImage(java.lang.String, java.lang.String, java.io.InputStream)
     */
    @Override
    public void cacheImage(final String imageKey, final InputStream srcImageStm, final String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace("IC #cacheImage called for imageKey: " + imageKey, context);

        ImageConverterStatus status = (isNotEmpty(imageKey) && (null != srcImageStm)) ?
            ImageConverterStatus.OK :
                ImageConverterStatus.KEY_NOT_VALID;

        try {
            if (ImageConverterStatus.OK == status) {
                m_imageConversionQueue.cache(imageKey, srcImageStm, implGetContext(context), null);
            }
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while caching Image", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.CACHE, requestStartTime, status);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#getImage(java.lang.String, java.lang.String, java.lang.String, java.io.InputStream)
     */
    @Override
    public InputStream cacheAndGetImage(final String imageKey, final String requestFormat, final InputStream srcImageStm, final String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace(new StringBuilder("IC #cacheAndGetImage called (imageKey / requestFormat): ").
            append(imageKey).append(" / ").append(requestFormat).toString(),
            context);

        ImageConverterStatus status = isNotEmpty(imageKey) ?
            ImageConverterStatus.OK :
                ImageConverterStatus.KEY_NOT_VALID;

        try {
            if (ImageConverterStatus.OK == status) {
                final String usedContext = implGetContext(context);

                try (final MetadataImage metadataImage = (null != srcImageStm) ?
                    m_imageConversionQueue.cacheAndGetMetadataImage(imageKey, srcImageStm, requestFormat, usedContext, null) :
                        (isNotEmpty(requestFormat) ?
                            m_imageConversionQueue.getMetadataImage(imageKey, requestFormat, usedContext) :
                                null)) {

                    InputStream ret = (null != metadataImage) ?
                        metadataImage.getImageInputStream(true) :
                            null;

                    if (null == ret) {
                        status = ImageConverterStatus.KEY_NOT_AVAILABLE;
                    }

                    return ret;
                }
            }
        } catch (IOException | ImageConverterException e) {
            LOG.trace("IC exception while caching and getting Image" ,e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw new ImageConverterException("IC error while caching and getting image: " + imageKey, e);
        } finally {
            implFinishRequest(RequestType.CACHE_AND_GET, requestStartTime, status);
        }

        return null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#cacheAndGetImageAndMetadata(java.lang.String, java.lang.String, java.io.InputStream, java.lang.String[])
     */
    @Override
    public MetadataImage cacheAndGetImageAndMetadata(String imageKey, String requestFormat, InputStream srcImageStm, String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace(new StringBuilder("IC #cacheAndGetImageAndMetadata called (imageKey / requestFormat): ").
            append(imageKey).append(" / ").append(requestFormat).toString(),
            context);

        ImageConverterStatus status = isNotEmpty(imageKey) ?
            (((null != srcImageStm) || isNotEmpty(requestFormat)) ? ImageConverterStatus.OK : ImageConverterStatus.KEY_NOT_AVAILABLE) :
                ImageConverterStatus.KEY_NOT_VALID;

        try {
            if (ImageConverterStatus.OK == status) {
                final String usedContext = implGetContext(context);

                return (null != srcImageStm) ?
                    m_imageConversionQueue.cacheAndGetMetadataImage(imageKey, srcImageStm, requestFormat, usedContext, null) :
                        m_imageConversionQueue.getMetadataImage(imageKey, requestFormat, usedContext);
            }
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while caching and getting ImageAndMetadata", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.CACHE_AND_GET, requestStartTime, status);
        }

        return null;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#clearImages()
     */
    @Override
    public void clearImages(final String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace("IC #clearImages called", context);

        ImageConverterStatus status = ImageConverterStatus.OK;

        try {
            remove(getAvailable(implCreateContextProperties(context)));
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while clearing Images", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.ADMIN, requestStartTime, status);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#clearImagesByKey(java.lang.String)
     */
    @Override
    public void clearImagesByKey(String imageKey) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace("IC #clearImagesByKey called for imageKey: " + imageKey);

        ImageConverterStatus status = isNotEmpty(imageKey) ?
            ImageConverterStatus.OK :
                ImageConverterStatus.KEY_NOT_VALID;

        try {
            if (ImageConverterStatus.OK == status) {
                remove(imageKey);
            }
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while clearing Images by key", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.ADMIN, requestStartTime, status);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#getKeyCount()
     */
    @Override
    public long getKeyCount(final String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace("IC #getKeyCount called", context);

        ImageConverterStatus status = ImageConverterStatus.OK;

        try {
            return isEmpty(context) ?
                getKeyCount() :
                    getAvailable(implCreateContextProperties(context)).length;
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while getting KeyCount", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.ADMIN, requestStartTime, status);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#getKeysByContext(java.lang.String)
     */
    @Override
    public String[] getKeys(final String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace("IC #getKeys called", context);

        ImageConverterStatus status = ImageConverterStatus.OK;

        try {
            return getAvailable(implCreateContextProperties(context));
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while getting Keys", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.ADMIN, requestStartTime, status);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.imageconverter.api.IImageConverter#getKeyCount()
     */
    @Override
    public long getTotalImagesSize(final String... context) throws ImageConverterException {
        final long requestStartTime = System.currentTimeMillis();

        implTrace("IC #getTotalImagesSize called", context);

        ImageConverterStatus status = ImageConverterStatus.OK;

        try {
            return isEmpty(context) ?
                getTotalImagesSize() :
                    getAvailableImagesSize(implCreateContextProperties(context));
        } catch (ImageConverterException e) {
            LOG.trace("IC exception while getting TotalImagesSize", e);

            status = ImageConverterStatus.GENERAL_ERROR;
            throw e;
        } finally {
            implFinishRequest(RequestType.ADMIN, requestStartTime, status);
        }
    }

    // - Interface -------------------------------------------------------------

    /**
     * @return
     */
    ImageFormat[] getImageFormats() {
        return m_imageFormats;
    }

    /**
     * @return
     */
    IFileItemService getFileItemService() {
        return m_fileItemService;
    }

    /**
     * @return
     */
    IMetadataReader getMetadataReader() {
        return m_metadataReader;
    }

    /**
     * @return
     * @throws ImageConverterException
     */
    long getKeyCount() throws ImageConverterException {
        try {
            return m_fileItemService.getSubGroupCount(ImageConverterUtils.IMAGECONVERTER_GROUPID);
        } catch (FileItemException e) {
            throw new ImageConverterException("IC error while getting image key count", e);
        }
    }

    /**
     * @return
     */
    long getTotalImagesSize() throws ImageConverterException {
        try {
            return m_fileItemService.getGroupLength(ImageConverterUtils.IMAGECONVERTER_GROUPID);
        } catch (FileItemException e) {
            throw new ImageConverterException("IC error while getting total images size", e);
        }
    }

    /**
     * @param searchProperties
     * @return
     * @throws ImageConverterException
     */
    long getAvailableImagesSize(@Nullable final Properties searchProperties) throws ImageConverterException {
        try {
            return m_fileItemService.getGroupLength(ImageConverterUtils.IMAGECONVERTER_GROUPID, searchProperties);
        } catch (FileItemException e) {
            throw new ImageConverterException("IC error while getting available images size", e);
        }
    }

    /**
     * @param imageKey
     * @return
     * @throws ImageConverterException
     */
    boolean isAvailable(@NonNull final String imageKey) throws ImageConverterException {
        try {
            return m_fileItemService.containsSubGroup(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey);
        } catch (FileItemException e) {
            throw new ImageConverterException("IC error while checking availability: " + imageKey, e);
        }
    }

    /**
     * @param searchProperties
     * @return
     * @throws ImageConverterException
     */
    @NonNull String[] getAvailable(@Nullable final Properties searchProperties) throws ImageConverterException {
        try {
            final Object[] foundObjs = (null != searchProperties) ?
                m_fileItemService.get(ImageConverterUtils.IMAGECONVERTER_GROUPID, searchProperties) :
                    m_fileItemService.getSubGroups(ImageConverterUtils.IMAGECONVERTER_GROUPID);

            if (isNotEmpty(foundObjs)) {
                final String[] ret = new String[foundObjs.length];
                final Set<String> imageKeySet = (null != searchProperties) ? new HashSet<>() : null;
                int count = 0;

                for (final Object curObj : foundObjs) {
                    if (null != curObj) {
                        final String curImageKey = (curObj instanceof IFileItem) ?
                            ((IFileItem) curObj).getSubGroupId() :
                                curObj.toString();

                        if (isNotEmpty(curImageKey) && ((null == imageKeySet) || imageKeySet.add(curImageKey))) {
                            ret[count++] = curImageKey;
                        }
                    }
                }

                return (String[]) ArrayUtils.subarray(ret, 0, count);
            }
        } catch (FileItemException e) {
            throw new ImageConverterException("IC error while retrieving available image keys", e);
        }

        return ArrayUtils.EMPTY_STRING_ARRAY;
    }

    /**
     * @param imageKey
     * @param srcImageStm
     * @param context
     * @return
     */
    void remove(@NonNull final String imageKey) throws ImageConverterException {
        ElementLocker.lock(imageKey, ElementLock.LockMode.WAIT_IF_PROCESSED);

        try {
            m_fileItemService.removeSubGroup(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey);
        } catch (FileItemException e) {
            throw new ImageConverterException("IC error while removing image key: " + imageKey, e);
        } finally {
            ElementLocker.unlock(imageKey);
        }
    }

    /**
     * @param imageKeys
     * @throws ImageConverterException
     */
    void remove(final String[] imageKeys) throws ImageConverterException {
        if (isNotEmpty(imageKeys) && !isTerminated()) {
            final String[] targetImageKeys = Arrays.asList(imageKeys).stream().
                filter(p -> ElementLocker.lock(p, ElementLock.LockMode.TRY_LOCK)).
                toArray(p -> new String[p]);

            if (isNotEmpty(targetImageKeys)) {
                try {
                    m_fileItemService.removeSubGroups(ImageConverterUtils.IMAGECONVERTER_GROUPID, targetImageKeys);
                } catch (FileItemException e) {
                    throw new ImageConverterException("IC error while removing image " + targetImageKeys.length + " keys", e);
                } finally {
                    Arrays.asList(targetImageKeys).forEach(p -> ElementLocker.unlock(p));
                }
            }
        }
    }

    /**
     * @param imageKeys
     * @throws ImageConverterException
     */
    void remove(@NonNull final Collection<String> imageKeysCollection) throws ImageConverterException {
        if (!imageKeysCollection.isEmpty() && !isTerminated()) {
            final String[] targetImageKeys = imageKeysCollection.stream().
                filter(p -> ElementLocker.lock(p, ElementLock.LockMode.TRY_LOCK)).
                toArray(p -> new String[p]);

            if (isNotEmpty(targetImageKeys)) {
                try {
                    m_fileItemService.removeSubGroups(ImageConverterUtils.IMAGECONVERTER_GROUPID, targetImageKeys);
                } catch (FileItemException e) {
                    throw new ImageConverterException("IC error while removing image " + targetImageKeys.length + " keys", e);
                } finally {
                    Arrays.asList(targetImageKeys).forEach(p -> ElementLocker.unlock(p));
                }
            }
        }
    }


    // - Implementation --------------------------------------------------------

    private static void implFinishRequest(@NonNull final RequestType requestType, final long requestStartTime, @NonNull final ImageConverterStatus imageConverterStatus) {
        final long requestDuration = System.currentTimeMillis() - requestStartTime;

        implTrace(new StringBuilder("IC request type ").
            append(requestType.name()).
            append(" finished in ").append(requestDuration).append(" ms").toString());

        ImageConverterUtils.IC_MONITOR.incrementRequests(requestType, requestDuration, imageConverterStatus);
    }

    /**
     * @param context
     * @return
     */
    private static Properties implCreateContextProperties(final String... context) {
        final String usedContext = implGetContext(context);
        Properties ret = null;

        if (null != usedContext) {
            (ret = new Properties()).setProperty(ImageConverterUtils.IMAGECONVERTER_KEY_CONTEXT, implGetContext(context));
        }

        return ret;
    }

    /**
     * @param context
     * @return
     */
    private static String implGetContext(final String... context) {
        return isNotEmpty(context) && isNotEmpty(context[0]) ? context[0] : null;
    }

    /**
     * @param traceMsg
     * @param context
     */
    private static void implTrace(final String traceMsg, final String... context) {
    	// TODO
        if (LOG.isTraceEnabled() && (null != traceMsg)) {
            LOG.trace(isNotEmpty(context) ?
                new StringBuilder(traceMsg).append( "(Context: ").append(context[0]).append(")").toString() :
                    traceMsg);
        }
    }

    // - Members ---------------------------------------------------------------

    final private IFileItemService m_fileItemService;

    final private IMetadataReader m_metadataReader;

    final private ImageFormat[] m_imageFormats;

    final private ImageConverterQueue m_imageConversionQueue;

    final private ImageConverterCache m_keyManager;

    final private AtomicBoolean m_terminated = new AtomicBoolean(false);
}
