/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.osgi;

import java.io.File;
import java.net.URI;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import org.apache.commons.collections.BidiMap;
import org.apache.commons.collections.bidimap.DualHashBidiMap;
import org.apache.commons.lang.StringUtils;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.config.ConfigurationService;
import com.openexchange.database.DatabaseService;
import com.openexchange.database.migration.DBMigrationMonitorService;
import com.openexchange.exception.OXException;
import com.openexchange.fileitem.impl.FileItemDatabase;
import com.openexchange.fileitem.impl.FileItemService;
import com.openexchange.fileitem.impl.FileItemUtils;
import com.openexchange.fileitem.impl.Services;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.filestore.FileStorage;
import com.openexchange.filestore.FileStorageService;
import com.openexchange.filestore.FileStorages;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.user.UserService;

/**
 * {@link FileItemServiceActivator}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemServiceActivator extends HousekeepingActivator {

    final private static String SERVICE_NAME = "FileItemService";

    /* (non-Javadoc)
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] {
            ConfigurationService.class,
            FileStorageService.class,
            ManagedFileManagement.class,
            UserService.class,
            DBMigrationMonitorService.class,
            DatabaseService.class,
        };
    }

    /* (non-Javadoc)
     * @see com.openexchange.osgi.HousekeepingActivator#start(org.osgi.framework.BundleContext)
     */
    @Override
    public void start(BundleContext ctx) throws Exception {
        super.start(ctx);
    }

    /* (non-Javadoc)
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        FileItemUtils.logInfo("starting bundle: " + SERVICE_NAME);

        Services.setServiceLookup(this);

        m_dbMigrationMonitorTracker = new ServiceTracker<>(context, DBMigrationMonitorService.class, new DBMigrationMonitorCustomizer(this, context));
        m_dbMigrationMonitorTracker.open();

        openTrackers();

        FileItemUtils.logInfo("successfully started bundle: " + SERVICE_NAME);
    }

    /* (non-Javadoc)
     * @see com.openexchange.osgi.HousekeepingActivator#stopBundle()
     */
    @Override
    protected void stopBundle() throws Exception {
        FileItemUtils.logInfo("stopping bundle: " + SERVICE_NAME);

        if (null != m_fileItemService) {
            m_fileItemService.shutdown();
        }

        closeTrackers();
        unregisterServices();
        Services.setServiceLookup(null);

        FileItemUtils.logInfo("successfully stopped bundle: " + SERVICE_NAME);
    }

    /**
     *
     */
    @SuppressWarnings("unused")
    protected void implRegisterService() {
        final ConfigurationService configService = getService(ConfigurationService.class);

        FileItemUtils.logInfo("Registering FileItemService");

        if (null != configService) {
            final long retryEndTimeMillis = System.currentTimeMillis() + 60000;
            final long retryTimeoutMillis = 1000;
            BidiMap fileStoreMap = null;

            // try for a maximum of 60 seconds to retrieve the filestores;
            // polling is necessary due to not deterministic initialization of
            // the FileStorageInfoService with delayed setting of internal services
            while ((null == (fileStoreMap = implGetFileStores(configService))) && (System.currentTimeMillis() < retryEndTimeMillis)) {
                try {
                    Thread.sleep(retryTimeoutMillis);
                } catch (InterruptedException e) {
                    // ok, try again until condition is met
                }
            }

            FileItemUtils.setSpoolDir(new File(configService.getProperty("com.openexchange.fileItem.spoolPath", "/tmp")));

            if ((null != fileStoreMap) && (fileStoreMap.size() > 0)) {
                try {
                    registerService(IFileItemService.class, m_fileItemService = new FileItemService(fileStoreMap, new FileItemDatabase(configService)));
                } catch(Throwable e) {
                    FileItemUtils.logExcp(e);
                }
            } else if (FileItemUtils.isLogError()) {
                FileItemUtils.logError("No valid FileStore service interfaces found => FileItem service will not be available");
            }
        }

        if (null == getFileItemService()) {
            FileItemUtils.logError("Could not register FileItemService");
        } else {
            FileItemUtils.logInfo("Successfully registered FileItemService");
        }
    }

    /**
     * @param configService
     * @return
     */
    private BidiMap implGetFileStores(@NonNull final ConfigurationService configService) {
        final FileStorageService fileStoreService = getService(FileStorageService.class);
        BidiMap ret = new DualHashBidiMap();

        if (null != fileStoreService) {
            final Set<String> usedStoreIdSet = new HashSet<>();
            int storeIndex = 0;

            for (String curStoreIdStr : configService.getProperty("com.openexchange.fileItem.fileStoreIds", "-1").split(",")) {
                curStoreIdStr = StringUtils.trim(curStoreIdStr);

                // don't add a FileStore with the same id more than once
                if (!usedStoreIdSet.contains(curStoreIdStr)) {
                    try {
                        final int curStoreId = Integer.valueOf(curStoreIdStr).intValue();

                        if (curStoreId > 0) {
                            try {
                                final URI fileStorageURI = FileStorages.getFullyQualifyingUriFor(curStoreId, FILEITEM_STORE_PREFIX + ++storeIndex);
                                final FileStorage fileStorage = (null != fileStorageURI) ? fileStoreService.getFileStorage(fileStorageURI) : null;

                                if ((null != ret) && (null != fileStorage)) {
                                    ret.put(fileStorage, Integer.valueOf(curStoreId));
                                    usedStoreIdSet.add(curStoreIdStr);
                                }
                            } catch (IllegalStateException e) {
                                // this exception is thrown, if the FileStorageInfoService has been
                                // started, but the internal members have not been set accordingly
                                // right now => return null to indicate this situation in order
                                // to try again later in time
                                ret = null;

                                FileItemUtils.logTrace("FileStorages#getFullyQualifyingUriFor is not yet able to successfully finish: ", Throwables.getRootCause(e));
                            } catch (OXException e) {
                                FileItemUtils.logError(new StringBuilder("Could not retrieve FileStore for given config id: ").
                                    append(curStoreId).
                                    append("(Reason: " ).append(Throwables.getRootCause(e).getMessage()).append(')').
                                    toString());
                            }
                        }
                    } catch (NumberFormatException e) {
                        FileItemUtils.logError(new StringBuilder("Provided store id is not valid: ").
                            append(curStoreIdStr).
                            append("(Reason: " ).append(Throwables.getRootCause(e).getMessage()).append(')').
                            toString());
                    }
                }
            }

            if ((null != ret) && (ret.size() < 1)) {
                FileItemUtils.logError("FileStoreService is available, but no valid FileStore " +
                    "could be found at all. Please check, if a FileStore has been correctly " +
                    "registered via 'registerfilestore' and at least one FileStore registration " +
                    "number has been correctly applied to the 'com.openexchange.fileItem.fileStoreIds' property");
            }
        } else {
            FileItemUtils.logError("Could not retrieve FileStoreService interface");
        }

        return ret;
    }

    /**
     *
     */
    protected void implUnregisterService() {
        unregisterService(IFileItemService.class);
    }

    /**
     * @return
     */
    protected IFileItemService getFileItemService() {
        return m_fileItemService;
    }

    // - Members ---------------------------------------------------------------


    private final static String FILEITEM_STORE_PREFIX = "fis";


    private ServiceTracker<DBMigrationMonitorService, DBMigrationMonitorService> m_dbMigrationMonitorTracker;

    private FileItemService m_fileItemService = null;

    // - Inner classes ---------------------------------------------------------

    /**
     * {@link DBMigrationMonitorCustomizer}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.0
     */
    private static class DBMigrationMonitorCustomizer implements ServiceTrackerCustomizer<DBMigrationMonitorService, DBMigrationMonitorService> {

        FileItemServiceActivator m_parent;

        private final BundleContext m_context;

        /**
         * Initializes a new {@link DBMigrationMonitorCustomizer}.
         * @param parent
         * @param context
         */
        DBMigrationMonitorCustomizer(final FileItemServiceActivator parent, final BundleContext context) {
            m_parent = parent;
            m_context = context;
        }

        /* (non-Javadoc)
         * @see org.osgi.util.tracker.ServiceTrackerCustomizer#addingService(org.osgi.framework.ServiceReference)
         */
        @Override
        public DBMigrationMonitorService addingService(ServiceReference<DBMigrationMonitorService> reference) {
            if (null != reference) {
                final DBMigrationMonitorService migrationMonitor = m_context.getService(reference);

                if (migrationMonitor != null) {
                    Executors.newSingleThreadExecutor().execute(new Runnable() {

                        @Override
                        public void run() {
                            try {
                                boolean dbUpdateInProgress = !migrationMonitor.getScheduledFiles().isEmpty();
                                if (dbUpdateInProgress) {
                                    long waitNanos = TimeUnit.SECONDS.toNanos(1L); // 1 second
                                    do {
                                        LockSupport.parkNanos(waitNanos);
                                        dbUpdateInProgress = !migrationMonitor.getScheduledFiles().isEmpty();
                                    } while (dbUpdateInProgress);
                                }
                            } catch (@SuppressWarnings("unused") Exception e) {
                                // ok
                            }

                            m_parent.implRegisterService();
                        }
                    });

                    return migrationMonitor;
                }

                m_context.ungetService(reference);
            }

            return null;
        }

        /* (non-Javadoc)
         * @see org.osgi.util.tracker.ServiceTrackerCustomizer#modifiedService(org.osgi.framework.ServiceReference, java.lang.Object)
         */
        @Override
        public void modifiedService(ServiceReference<DBMigrationMonitorService> reference, DBMigrationMonitorService service) {
            //
        }

        /* (non-Javadoc)
         * @see org.osgi.util.tracker.ServiceTrackerCustomizer#removedService(org.osgi.framework.ServiceReference, java.lang.Object)
         */
        @Override
        public void removedService(ServiceReference<DBMigrationMonitorService> reference, DBMigrationMonitorService service) {
            m_context.ungetService(reference);
        }

    }
}
