/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.client;

import static org.apache.commons.lang.StringUtils.indexOfIgnoreCase;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.lastIndexOfIgnoreCase;
import java.io.BufferedReader;
import java.io.StringReader;
import java.net.URL;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.ImageServerUtil;
import com.openexchange.imageconverter.client.generated.modules.ImageConverterApi;

/**
 * {@link ImageConverterRemoteValidator}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
class ImageConverterRemoteValidator {

    /**
     * ESTABLISH_INITIAL_CONNECTION_DELAY_MILLIS
     */
    final protected static int ESTABLISH_INITIAL_CONNECTION_DELAY_MILLIS = 10000;

    /**
     * ESTABLISH_INITIAL_CONNECTION_PERIOD_MILLIS
     */
    final protected static int ESTABLISH_INITIAL_CONNECTION_PERIOD_MILLIS = 3000;

    /**
     * ESTABLISH_INITIAL_CONNECTION_TIMEOUT_MILLIS
     */
    final protected static int ESTABLISH_INITIAL_CONNECTION_TIMEOUT_MILLIS = 10000;

    /**
     * CHECK_REMOTE_CONNECTION_TIMEOUT_MILLIS
     */
    final protected static int CHECK_REMOTE_CONNECTION_TIMEOUT_MILLIS = 20000;

    /**
     * ENABLE_LOG_PERIOD_MILLIS
     */
    final protected static long ENABLE_LOG_PERIOD_MILLIS = 60000;

    /**
     * {@link ImageConverterServerValidatorCallable}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.0
     */
    protected class ImageConverterServerValidatorCallable implements Callable<Integer> {

        /**
         * Initializes a new {@link ImageConverterServerValidatorCallable}.
         */
        protected ImageConverterServerValidatorCallable() {
            super();
        }

        /**
         * @return
         * @throws Exception
         */
        @Override
        public Integer call() throws Exception {
            final Logger LOG = ImageConverterClient.getLogger();
            final ImageConverterApi imageConverterApi = m_imageConverterApiProvider.acquire();
            int remoteApiVersion = 0;
            String statusResponse = null;

            if (null != imageConverterApi) {
                try {
                    if (isNotEmpty(statusResponse = imageConverterApi.status())) {
                        statusResponse = statusResponse.toLowerCase();

                        try (final BufferedReader stringReader = new BufferedReader(new StringReader(statusResponse))) {
                            for (String readLine = null; (readLine = stringReader.readLine()) != null;) {
                                int indexOf = 0;

                                if ((remoteApiVersion <= 0) &&
                                    ((indexOf = indexOfIgnoreCase(readLine, "id:")) > -1) &&
                                    ((indexOf = indexOfIgnoreCase(readLine.substring(indexOf + 3), ImageServerUtil.IMAGE_SERVER_ID)) > -1)) {

                                    remoteApiVersion = 1;
                                }

                                if ((indexOf = lastIndexOfIgnoreCase(readLine, "api:")) > -1) {

                                    final Matcher matcher = API_VERSION_PATTERN.matcher(readLine.substring(indexOf));

                                    if (matcher.find()) {
                                        remoteApiVersion = Integer.valueOf(matcher.group(1)).intValue();
                                    }
                                }
                            }
                       }
                    }
                } catch (Throwable e) {
                    LOG.trace("IC client RemoteValidator not able to get remote status of server", e);
                } finally {
                    // disconnect and try again, if no valid answer has been received so far
                    if (remoteApiVersion < 1) {
                        if (m_remoteAPIVersion.get() > 0) {
                            implSetConnectionLost();
                        } else {
                            synchronized (self) {
                                final long curLogTime = System.currentTimeMillis();

                                if ((0 == m_lastLogTime) || ((curLogTime - m_lastLogTime) >= ENABLE_LOG_PERIOD_MILLIS)) {
                                    LOG.error("IC client remote connection could not be established => please check remote ImageConverter server setup at: {}", m_remoteUrl);
                                    m_lastLogTime = curLogTime;
                                }
                            }
                        }
                    }

                    m_imageConverterApiProvider.release(imageConverterApi);
                }
            }

            synchronized (self) {
                if ((remoteApiVersion > 0) && (m_remoteAPIVersion.get() < 1)) {
                    LOG.info("IC client established remote connection to ImageConverter server at: {}", m_remoteUrl);
                }

                m_remoteAPIVersion.set(remoteApiVersion);
                m_isRunning.compareAndSet(true, false);
            }

            return Integer.valueOf(remoteApiVersion);
        }
    }

    /**
     * @param remoteUrl
     */
    ImageConverterRemoteValidator(@NonNull final ImageConverterClientConfig clientConfig, @NonNull final ImageConverterApiProvider apiProvider) {
        super();

        m_imageConverterClientConfig = clientConfig;
        m_imageConverterApiProvider  = apiProvider;

        if (null != (m_remoteUrl = m_imageConverterClientConfig.getImageConverterServerURL())) {
            m_callable = new ImageConverterServerValidatorCallable();

            // initial trigger at end of construction
            implEstablishInitialConnection(ESTABLISH_INITIAL_CONNECTION_TIMEOUT_MILLIS);
        } else {
            m_callable = null;
        }
    }

    /**
     * @return
     */
    ImageConverterClientConfig getConfig() {
        return m_imageConverterClientConfig;
    }

    /**
     * @return
     */
    URL getRemoteUrl() {
        return m_remoteUrl;
    }

    /**
     * @return
     */
    boolean isRemoteServerSupported() {
        return (null != m_callable);
    }

    /**
     * @return
     */
    boolean isConnected() {
        return (getRemoteAPIVersion() >= 1);
    }

    /**
     * @return
     */
    synchronized int getRemoteAPIVersion() {
        if (isRemoteServerSupported() && (m_remoteAPIVersion.get() < 1) && m_isRunning.compareAndSet(false, true)) {
            m_executorService.submit(new FutureTask<>(m_callable));
        }

        return m_remoteAPIVersion.get();
    }

    /**
     *
     */
    synchronized void resetConnection() {
        if ((null != m_callable) && (m_remoteAPIVersion.get() >= 1) && !m_isRunning.get()) {
            implSetConnectionLost();
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param timeoutMillis
     */
    protected void implEstablishInitialConnection(final long timeoutMillis) {

        m_executorService.submit(new Callable<Boolean>() {

            @Override
            public Boolean call() throws Exception {
                Thread.sleep(Math.max(0, ESTABLISH_INITIAL_CONNECTION_DELAY_MILLIS - ESTABLISH_INITIAL_CONNECTION_PERIOD_MILLIS));

                final long endTryTimeMillis = System.currentTimeMillis() + timeoutMillis;
                boolean ret = false;

                do {
                    Thread.sleep(ESTABLISH_INITIAL_CONNECTION_PERIOD_MILLIS);
                    ret = (getRemoteAPIVersion() >= 1);
                } while (!ret && (System.currentTimeMillis() < endTryTimeMillis));

                return Boolean.valueOf(ret);
            }
        });
    }

    /**
     *
     */
    protected void implSetConnectionLost() {
        m_remoteAPIVersion.set(-1);

        ImageConverterClient.getLogger().error("IC client lost remote connection to remote URL: {}", m_remoteUrl);
    }

    // - Static members --------------------------------------------------------

    final protected static Pattern API_VERSION_PATTERN = Pattern.compile("[aA][pP][iI]\\:\\s*v([0-9]+)");

    // - Members ---------------------------------------------------------------

    final protected ImageConverterRemoteValidator self = this;

    final protected ExecutorService m_executorService = Executors.newCachedThreadPool();

    final protected AtomicBoolean m_isRunning = new AtomicBoolean(false);

    final protected AtomicInteger m_remoteAPIVersion = new AtomicInteger(-1);

    final protected ImageConverterClientConfig m_imageConverterClientConfig;

    final protected ImageConverterApiProvider m_imageConverterApiProvider;

    final protected URL m_remoteUrl;

    final protected ImageConverterServerValidatorCallable m_callable;

    protected long m_lastLogTime = 0;
}
