/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.client;

import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.io.FileUtils;
import com.openexchange.annotation.NonNull;
import com.openexchange.config.ConfigurationService;

/**
 * {@link ImageConverterClientConfig}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public final class ImageConverterClientConfig {

    final private static File DEFAULT_TEMP_DIR = new File("/tmp");

    final private static String REMOTE_TEST_URL = "file:///ImageConverterClient_TestURL";

    final private static String IMAGECONVERTER_CLIENT_TEMP_DIRNAME = "oxiccli";

    /**
     * Initializes a new {@link ImageConverterClientConfig}.
     */
    @SuppressWarnings("unused")
    private ImageConverterClientConfig() {
        // not to be used
    }

    /**
     * Initializes a new {@link ImageConverterClientConfig}.
     * @param configurationService
     */
    public ImageConverterClientConfig(@NonNull final ConfigurationService configurationService) {
        super();

        m_configurationService = configurationService;
        implReadConfig();
    }

    /**
     *
     */
    public void shutdown() {
        if (m_terminated.compareAndSet(false, true)) {
            FileUtils.deleteQuietly(m_spoolDir);
        }
    }

    /**
     * @return
     */
    public boolean isValid() {
        return (null != m_imageConverterServerURL);
    }

    /**
     * @return
     */
    public boolean isTest() {
        boolean ret = false;

        try {
            ret =(isValid() && m_imageConverterServerURL.equals(new URL(REMOTE_TEST_URL)));
        } catch (@SuppressWarnings("unused") MalformedURLException e) {
            // ok, should not happen with check of test Url
        }

        return ret;
    }

    /**
     * @return
     */
    public URL getImageConverterServerURL() {
        return m_imageConverterServerURL;
    }

    /**
     * @return
     */
    public File getSpoolDir() {
        return (!m_terminated.get() ? m_spoolDir : DEFAULT_TEMP_DIR);
    }

    // - Implementation --------------------------------------------------------

    /**
     *
     */
    private void implReadConfig() {
        String curValue = null;

        if (isNotEmpty(curValue= m_configurationService.getProperty("com.openexchange.imageconverter.client.remoteImageConverterUrl"))) {
            try {
                if (REMOTE_TEST_URL.equals(curValue)) {
                    m_imageConverterServerURL = new URL(REMOTE_TEST_URL);
                } else {
                    m_imageConverterServerURL = new URL(curValue);
                }
            } catch (@SuppressWarnings("unused") Exception e) {
                m_imageConverterServerURL = null;
            }
        }

        implSetSpoolDir(new File(m_configurationService.getProperty("com.openexchange.imageconverter.client.spoolPath", "/tmp")));
    }

    /**
     * @param spoolDir
     */
    private void implSetSpoolDir(@NonNull final File spoolDir) {
        if (null != spoolDir) {
            File targetDir = new File(spoolDir, IMAGECONVERTER_CLIENT_TEMP_DIRNAME);

            // use or create given spool directory, use /tmp as fallback parent directory
            if ((targetDir.exists() && targetDir.canWrite() && targetDir.isDirectory()) ||
                targetDir.mkdirs() || (targetDir = new File("/tmp", IMAGECONVERTER_CLIENT_TEMP_DIRNAME)).mkdirs()) {

                try {
                    FileUtils.cleanDirectory(m_spoolDir = targetDir);
                } catch (@SuppressWarnings("unused") IOException e) {
                    ImageConverterClient.getLogger().error("IC client is not able to delete spool directory when shut down: {}", m_spoolDir.toString());
                }
            }
         }
    }

    // - Members ---------------------------------------------------------------

    final private AtomicBoolean m_terminated = new AtomicBoolean(false);

    private ConfigurationService m_configurationService = null;

    private URL m_imageConverterServerURL = null;

    private File m_spoolDir = new File("/tmp");
}
