/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.osgi;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import org.apache.commons.io.IOUtils;
import org.osgi.framework.ServiceReference;
import com.google.common.base.Throwables;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.IImageConverter;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.impl.ImageConverter;
import com.openexchange.imageconverter.impl.ImageConverterConfig;
import com.openexchange.imageconverter.impl.ImageConverterMBean;
import com.openexchange.imageconverter.impl.ImageConverterUtils;
import com.openexchange.imageconverter.impl.Services;
import com.openexchange.management.ManagementService;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.osgi.SimpleRegistryListener;

//=============================================================================
public class ImageConverterActivator extends HousekeepingActivator {

    public ImageConverterActivator() {
        super();
    }

    //-------------------------------------------------------------------------

    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] {
            ConfigurationService.class,
            IFileItemService.class,
            IMetadataReader.class
        };
    }

    //-------------------------------------------------------------------------

    @Override
    protected void startBundle() throws Exception {
        try {
            LOG.info("starting bundle: " + SERVICE_NAME);

            Services.setServiceLookup(this);

            final ConfigurationService configService = getService(ConfigurationService.class);

            if (null == configService) {
                throw new ImageConverterException("Could not get Configuration service");
            }

            final IFileItemService fileItemService = getService(IFileItemService.class);

            if (null == fileItemService) {
                throw new ImageConverterException("Could not get IFileItemService service");
            }

            final IMetadataReader metadataReader = getService(IMetadataReader.class);

            if (null == metadataReader) {
                throw new ImageConverterException("Could not get IMetadataReader service");
            }

            m_imageConverterConfig = new ImageConverterConfig(configService);
            m_imageConverter = new ImageConverter(fileItemService, metadataReader, m_imageConverterConfig);

            registerService(IImageConverter.class, m_imageConverter);

            implRegisterMBean();

            openTrackers();

            LOG.info("successfully started bundle: " + SERVICE_NAME);
        } catch (Throwable e) {
            LOG.error("... starting bundle: " + SERVICE_NAME + " failed", Throwables.getRootCause(e));
            throw new RuntimeException(e);
        }
    }

    //-------------------------------------------------------------------------

    @Override
    protected void stopBundle() throws Exception {
        LOG.info("stopping bundle: " + SERVICE_NAME);

        closeTrackers();
        unregisterServices();
        Services.setServiceLookup(null);

        if (null != m_imageConverter) {
            ImageConverterUtils.IC_MONITOR.shutdown();

            m_imageConverter.shutdown();
            m_imageConverter = null;
        }

        if (null != m_imageConverterConfig) {
            IOUtils.closeQuietly(m_imageConverterConfig);
        }

        LOG.info("successfully stopped bundle: " + SERVICE_NAME);
    }

    // - Implementation --------------------------------------------------------

    /**
     * Registering the monitoring MBean once
     */
    private void implRegisterMBean() {
        if (null == m_imageConverterMBean) {
            final ObjectName mbeanObjectName = ImageConverterMBean.getObjectName();

            if (null != mbeanObjectName) {
                try {
                    m_imageConverterMBean = new ImageConverterMBean(ImageConverterUtils.IC_MONITOR);
                    ImageConverterUtils.IC_MONITORING = m_imageConverterMBean;

                    track(ManagementService.class, new SimpleRegistryListener<ManagementService>() {
                        @Override
                        public void added(ServiceReference<ManagementService> ref, ManagementService service) {
                            try {
                                service.registerMBean(mbeanObjectName, m_imageConverterMBean);
                                LOG.info("IC monitoring successfull registered");
                            } catch (final OXException e) {
                                LOG.error(e.getMessage(), Throwables.getRootCause(e));
                            }
                        }

                        @Override
                        public void removed(ServiceReference<ManagementService> ref, ManagementService service) {
                            try {
                                service.unregisterMBean(mbeanObjectName);
                                LOG.info("IC monitoring successfully unregistered");
                            } catch (final OXException e) {
                                LOG.error(e.getMessage(), Throwables.getRootCause(e));
                            }
                        }
                    });
                } catch (NotCompliantMBeanException e) {
                    LOG.error(e.getMessage(), Throwables.getRootCause(e));
                }
            } else {
                LOG.error("IC error when initializing monitoring MBean: invalid ObjectName");
            }
        } else {
            LOG.warn("IC monitoring MBean already inititalized");
        }
    }

    // - Static Members --------------------------------------------------------

    private static final String SERVICE_NAME = "Open-Xchange ImageConverter";

    // - Members ---------------------------------------------------------------

    protected ImageConverterConfig m_imageConverterConfig = null;

    protected ImageConverter m_imageConverter = null;

    protected ImageConverterMBean m_imageConverterMBean = null;
}
